<?php

namespace ThalamusSDK;

/**
 * Provides methods to connect with the Thalamus API
 *
 * @author Ezequiel
 * @version 2.0.1
 */
use \ThalamusSDK\Model\Person;
use \ThalamusSDK\Model\Company;
use \ThalamusSDK\Model\Activity;
use \ThalamusSDK\Model\Message;
use \ThalamusSDK\Model\Caso;

class Thalamus {
	/**
	 * @const string version de la SDK.
	 */
	const VERSION_SDK = '2.0.1';
	
	/**
	 * @const string version de la API
	 */
	const VERSION_API = TH_API_VERSION;
	
	/**
	 * @const string test|prod
	 */
	const THALAMUS_ENVIRONMENT = TH_ENVIRONMENT;
	
	/**
	 * @const client
	 */
	const THALAMUS_CLIENT = TH_CLIENT;
	
	/**
	 * @const string
	 */
	const THALAMUS_TOUCHPOINT = TH_TOUCHPOINT;
	
	/**
	 * @const string
	 */
	const THALAMUS_TOKEN = TH_TOKEN;
	
	/**
	 * @const default password para registros sin password.
	 */
	const DEFAULT_PASSWORD = 'default';
	
	/**
	 *
	 * @var ThalamusSession The session used for requests
	 */
	private $session;
	
	/**
	 * getMe returns Thalamus instance
	 *
	 * @return \ThalamusSDK\Thalamus
	 */
	public static function getInstance() {
		static $instance = null;
		if (null === $instance) {
			$instance = new static ();
		}
		
		return $instance;
	}
	
	/**
	 * Protected constructor to prevent creating a new instance of the
	 * *Singleton* via the `new` operator from outside of this class.
	 */
	protected function __construct() {
		$this->session = ThalamusSession::getSession ();
	}
	
	/**
	 * Private clone method to prevent cloning of the instance of the
	 * *Singleton* instance.
	 *
	 * @return void
	 */
	private function __clone() {
	}
	
	/**
	 * Private unserialize method to prevent unserializing of the *Singleton*
	 * instance.
	 *
	 * @return void
	 */
	private function __wakeup() {
	}

	/**
	 * setTags
	 *
	 * @param string $tags
	 * @return Thalamus object
	 */
	public function setTags($tags) {
		$this->session->setTags ( $tags );
		return $this;
	}

	/**
	 * setSource
	 *
	 * @param string $source
	 * @return Thalamus object
	 */
	public function setSource($source) {
		$this->session->setSource ( $source );
		return $this;
	}
	
	/**
	 * isAlive checks if thalamus server is alive
	 *
	 * @return boolean
	 */
	public static function isAlive() {
		self::getInstance ();
		return ThalamusRequest::get ( '/isAlive' )->execute ()->getResponse ()->isAlive == true;
	}
	
	/**
	 * login user
	 *
	 * @param string $principal        	
	 * @param string $password
	 *        	optional
	 * @param integer $documentType
	 *        	optional
	 * @return \ThalamusSDK\Model\Person
	 */
	public function loginPerson($principal, $password = "", $documentType = false) {
		$this->session->reset ();
		
		$params = array (
				'principal' => ($documentType) ? $documentType . '$' . $principal : $principal,
				'password' => ! empty ( $password ) ? $password : self::DEFAULT_PASSWORD
		);
		
		$queue = new ThalamusMultiRequest ();
		
		$queue->addRequest ( ThalamusRequest::post ( '/signin', $params )->noAuth () );
		$queue->addRequest ( ThalamusRequest::get ( '/person' )->auth ( $principal, $password ), \ThalamusSDK\Model\Person::className () );
		
		$responses = $queue->execute ();
		
		$this->session->setUser ( $params ['principal'] );
		$this->session->setAccessToken ( $params ['password'] );
		
		$person = $responses [1];
		
		$this->session->setPerson ( $person );
		
		return $person;
	}
	
	public function loginCompany($principal, $password = "", $documentType = false) {
		$this->session->reset ();
	
		$params = array (
				'principal' => ($documentType) ? $documentType . '$' . $principal : $principal,
				'password' => ! empty ( $password ) ? $password : self::DEFAULT_PASSWORD
		);
	
		$queue = new ThalamusMultiRequest ();
	
		$queue->addRequest ( ThalamusRequest::post ( '/signin', $params )->noAuth () );
		$queue->addRequest ( ThalamusRequest::get ( '/company' )->auth ( $principal, $password ), \ThalamusSDK\Model\Company::className () );
	
		$responses = $queue->execute ();
	
		$this->session->setUser ( $params ['principal'] );
		$this->session->setAccessToken ( $params ['password'] );
	
		$person = $responses [1];
	
		$this->session->setPerson ( $person );
	
		return $person;
	}
	
	/**
	 * getPerson
	 *
	 * @return Person
	 */
	public function getPerson($cache = true) {
		if ($cache)
			return ($this->session->getPerson ()) ? $this->session->getPerson () : ThalamusRequest::get ( '/person' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () );
		else
			return ThalamusRequest::get ( '/person' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () );
	}

	/**
	 * getPerson
	 *
	 * @return Person
	 */
	public function getCompany($cache = true) {
		if ($cache)
			return ($this->session->getPerson ()) ? $this->session->getPerson () : ThalamusRequest::get ( '/company' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Company::className () );
		else {
			return $this->session->setPerson( ThalamusRequest::get ( '/company' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Company::className () ) );
		}
			
	}
	
	/**
	 * logout
	 *
	 * @return ThalamusResponse
	 */
	public function logout() {
		$response = ThalamusRequest::get ( '/logout' )->execute ();
		
		$this->session->destroy ();
		
		return $response;
	}
	
	/**
	 * register person
	 *
	 * @param array $profile        	
	 * @param array $optIns        	
	 * @param string $principal        	
	 * @param string $password
	 *        	(optional)
	 * @param array $consumers
	 *        	(optional)
	 *        	
	 * @return \ThalamusSDK\Model\Person object
	 */
	public function register(Array $profile, Array $optIns, $principal, $password = self::DEFAULT_PASSWORD, Array $consumers = array()) {
		$params = array (
				'consumers' => $consumers,
				'optIns' => $optIns,
				'profile' => $profile,
				'credential' => array (
						'principal' => $principal,
						'password' => $password 
				) 
		);
		
		$response = ThalamusRequest::post ( '/person', $params )->noAuth ()->execute ();
		
		$this->session->setUser ( $principal );
		$this->session->setAccessToken ( $password );
		
		$person = $this->getPerson ( false );
		
		$this->session->setPerson ( $person );
		
		return $person;
	}
	
	/**
	 * requestResetPassword
	 *
	 * @param string $principal        	
	 * @param string $documentType        	
	 * @return ThalamusResponse object
	 */
	public function requestResetPassword($principal, $documentType = false) {
		if ($documentType !== false) {
			$principal = $documentType . ':' . $principal;
		}
		
		$params = array (
				"principal" => $principal 
		);
		
		return ThalamusRequest::post ( '/person/password/requestreset', $params )->noAuth ()->execute ();
	}
	
	/**
	 * resetPassword
	 *
	 * @param string $tokenLink        	
	 * @param string $newPassword        	
	 * @param string $principalLink        	
	 * @return ThalamusResponse object
	 */
	public function resetPassword($tokenLink, $newPassword, $principalLink) {
		$params = array (
				'token' => $tokenLink,
				'password' => $newPassword,
				'principal' => $principalLink
		);
		
		return ThalamusRequest::put ( '/person/password', $params )->noAuth ()->execute();
		
	}
	
	/**
	 * fireInteraction
	 *
	 * @param integer $interactionTypeCode  	
	 * @param string $additionalInfo        	
	 * @return ThalamusResponse object
	 */
	public function fireInteraction($interactionTypeCode, $additionalInfo = "") {
		$params = array (
				'interactionTypeCode' => $interactionTypeCode,
				'additionalInfo' => $additionalInfo 
		);
		
		return ThalamusRequest::post ( '/interaction/fire', $params )->execute ();
	}
	
	/**
	 *
	 * @param boolean $read        	
	 * @param int $page        	
	 * @param int $size        	
	 *
	 * @return array of Message objects
	 */
	public function getInBox($read = null, $page = null, $size = null) {
		return $this->getPerson ()->getInBox ( $read, $page, $size );
	}
	
	/**
	 *
	 * @param string $caseId        	
	 *
	 * @return string
	 */
	public function getCases($caseId = false) {
		if ($caseId)
			// return ThalamusRequest::get ( '/cases/' . $caseId )->execute ()->getThalamusModel ()->getProperty ( 'case', Caso::className () );
			return ThalamusRequest::get ( '/cases/' . $caseId )->execute ()->getThalamusModel ( Caso::className () );
		else
			return ThalamusRequest::get ( '/cases/' )->execute ()->getThalamusModelList ( Caso::className (), 'cases' );
	}
	
	/**
	 *
	 * @param string $categoryCode
	 * @param string $typeCode
	 * @param array  $caseData
	 * @param string $description
	 */
	public function createCase($categoryCode, $typeCode, Array $caseData = array(), $description = "") {
		$params = array (
				'categoryCode' => $categoryCode,
				'typeCode' => $typeCode 
		);
		
		if (! empty ( $caseData )) {
			$params ['caseData'] = $caseData;
		}
		
		if (! empty ( $description )) {
			$params ['comments'] = array (
				array (
						'description' => $description 
				) 
			);
		}
		
		return ThalamusRequest::post ( '/cases', $params )->auth($GLOBALS['user'])->execute ()->getThalamusModel ()->getProperty ( 'case', Caso::className () );
	}

	/**     	
	 *
	 * @return array
	 */
	public function getParties() {
		return ThalamusRequest::get ( '/operator/parties/' )->execute ()->getThalamusModelList ( Person::className (), 'partyProfile' );
	}

	/**     	
	 * @param int $caseId
	 * @param string $supervisor
	 * @return string
	 */
	public function assignCase($caseId = false, $operator = false, $operatorGroup = false) {
		if($operator){
			$params = array (
				'email' => $operator 
			);
		}else{
			$params = array (
				'groupName' => $operatorGroup 
			);
		}

		return ThalamusRequest::put ( '/cases/' .$caseId. '/assignment', $params )->auth($GLOBALS['user'])->execute ();
	}
	
	/**
	 *
	 * @param string $callbackUrl        	
	 */
	public function getFacebookLink($callbackUrl) {
		return ThalamusRequest::post ( '/signin/facebook?callback_url=' . $callbackUrl )->noAuth ()->execute ()->getResponse ()->link->href;
	}
	
	/**
	 *
	 * @param string $code        	
	 * @param string $callbackUrl
	 * @return unknown
	 */
	public function loginFacebook($code, $callbackUrl) {
		$response = ThalamusRequest::get ( "/signin/facebook?code=$code&callback_url=$callbackUrl" )->noAuth ()->execute ();
		
		// exito
		if ($response->getResponse ()->reason == "SuccessfullySignIn") {
			
			$this->session->setUser ( $response->getResponse ()->data->principal );
			$this->session->setAccessToken ( str_replace ( ':', '$', $response->getResponse ()->data->tmp_password ) );
			
			$person = $this->getPerson ( false );
			
			$this->session->setPerson ( $person );
			
			return $person;
			
			// si no esta registrado
		} else if ($response->getResponse ()->reason == "NotRegistered") {
			// Retorna exception con mensaje "NotRegistered"
			throw new ThalamusOtherException ( 'NotRegistered', $response->getResponse (), 200 );
		}
		
		return $response;
	}
	
	/**
	 *
	 * @param array $profile        	
	 * @param array $socialConnections        	
	 * @param array $optIns        	
	 * @param string $principal        	
	 * @param string $password        	
	 * @param array $consumers        	
	 * @return Person object
	 */
	public function registerFacebook(Array $profile, Array $facebookSocialConnection, Array $optIns, $principal, $password = self::DEFAULT_PASSWORD, Array $consumers = array()) {
		$params = array (
				'consumers' => $consumers,
				'profile' => $profile,
				'optIns' => $optIns,
				'socialConnections' => array (
						$facebookSocialConnection 
				) 
		);
		
		$response = ThalamusRequest::post ( '/person', $params )->noAuth ()->execute ();
		
		$this->session->setUser ( $response->getResponse ()->temporal_credential->principal );
		$this->session->setAccessToken ( str_replace ( ':', '$', $response->getResponse ()->temporal_credential->tmp_password ) );
		
		$person = $this->getPerson ( false );
		
		$this->session->setPerson ( $person );
		
		return $person;
	}
	
	/**
	 * getEvent
	 *
	 * @param string $eventCode        	
	 * @return \ThalamusSDK\Model\Activity\Event\Event
	 */
	public function getEvent($eventCode) {
		return $this->getPerson ()->getActivity ()->getEvent ( $eventCode );
	}
	
	/**
	 * getSurvey
	 *
	 * @param string $surveyCode        	
	 * @return \ThalamusSDK\Model\Activity\Survey\Survey
	 */
	public function getSurvey($surveyCode) {
		return $this->getPerson ()->getActivity ()->getSurvey ( $surveyCode );
	}

	/**
	 * getMileage
	 *
	 * @param string $mileageCode
	 * @return \ThalamusSDK\Model\Activity\Mileage\Mileage
	 */
	public function getMileage($mileageCode) {
		return $this->getPerson ()->getActivity ()->getMileage ( $mileageCode );
	}

	/**
	 *
	 * @param Number $size
	 * @param Number $page
	 * @return Chain object
	 */
	public function getChain($payItForwardCode, $size = null, $page = null) {
		$params = array (
			'size' => $size,
			'page' => $page
		);

		return ThalamusRequest::get ( "/payitfoward/$payItForwardCode/gifts/chain/public" . ThalamusHelper::paramsToUrl ( $params ) )
								->noAuth ()
								->execute ()
								->getThalamusModel ( \ThalamusSDK\Model\Activity\PayItForward\Chain::classname (), array ('activityCode' => $payItForwardCode) );
	}

	/**
	 * 
	 * @param string $payItForwardCode
	 * @return \ThalamusSDK\Model\Activity\PayItForward\PayItForward
	 */
	public function getPayItForward($payItForwardCode) {
		return $this->getPerson()->getActivity()->getPayItForward($payItForwardCode);
	}
	
	/**
	 *
	 * @param string $activityId
	 */
	public function activityLogin($activityId) {
		return ThalamusRequest::post ( "/activities/$activityId/login" )->execute ();
	}
	
	/**
	 * Returns test log.
	 * 
	 * @return string
	 */
	public static function test() {
		ThalamusTest::run();
	}
	
	/**
	 * 
	 * @return multitype:multitype:string NULL
	 */
	public function getPartnerSKUs() {
		return ThalamusRequest::get( "/partner/skus" )->execute()->getThalamusModelList( 'ThalamusSDK\ThalamusModel', 'partnerSkus');
	}
	
	/**
	 *
	 * @param string $code
	 * @return array
	 */
	public function getPartnerSku($code) {
		
		return ThalamusRequest::get( "/partner/skus/" . rawurlencode($code) )->execute()->getThalamusModel();
	}
	
	/**
	 * 
	 * @param string $skuCode
	 * @param string $code
	 * @param string $name
	 * @param string $equivalence
	 * @return multitype:string NULL
	 */
	public function createPartnerSKU($skuCode, $code, $name, $equivalence, $skuCodeKey) {
		
		$params = array(
			"skuCode" => $skuCode,
			"code" => $code,
			"name" => $name,
			"equivalence" => $equivalence,
			"skuCodeKey" => $skuCodeKey
		);
		
		$sku = ThalamusRequest::post( "/partner/skus", $params )->execute()->getThalamusModel()->getProperty('partner_sku');
		
		return array(
			'skuCode' => $sku->getProperty('skuCode'),
			'code' => $sku->getProperty('code'),
			'name' => $sku->getProperty('name'),
			'equivalence' => $sku->getProperty('equivalence'),
			'skuName' => $sku->getProperty( 'name' ),
			'brand' => 'McCain'
		);
		
	}
	
	/**
	 * 
	 * @param string $skuCode
	 * @param string $code
	 * @param string $name
	 * @param string $equivalence
	 * @return multitype:string NULL
	 */
	public function updatePartnerSKU($skuCode, $code, $name, $equivalence, $skuCodeKey) {
	
		$params = array(
				"skuCode" => $skuCode,
				"code" => $code,
				"name" => $name,
				"equivalence" => $equivalence,
				"skuCodeKey" => $skuCodeKey
		);
	
		$sku = ThalamusRequest::put( "/partner/skus/" . rawurlencode($code), $params )->execute()->getThalamusModel()->getProperty('partner_sku');
	
		return array(
				'skuCode' => $sku->getProperty('skuCode'),
				'code' => $sku->getProperty('code'),
				'name' => $sku->getProperty('name'),
				'equivalence' => $sku->getProperty('equivalence'),
				'skuName' => $sku->getProperty( 'name' ),
				'brand' => 'McCain'
		);
	
	}
	
	/**
	 * 
	 * @return array
	 */
	public function getSkus() {
		return ThalamusRequest::get( "/referencedata/skus" )->execute()->getResponse();
	}
	
	/**
	 * 
	 * @param string $file
	 * @param int $id
	 */
	public function importFile($file, $id) {
		
		$params = array(
			'file' => $file
		);
		
		return ThalamusRequest::post( "/cases/$id/files" , $params )->sendFile()->execute();
		
	}
	
	/**
	 * @param array $filter  
	    'type' => $type,
		'page' => $page,
		'pageSize' => $size,
		'orderBy' => $orderBy,
		'ascending' => $ascending,
		'name' => $name,
		'creationDate' => $creationDate,
		'state' => $state,
		'withErrors' => $withErrors
	 */
	public function getImports(Array $params = array()) {
		return ThalamusRequest::get ( "/import/status" . ThalamusHelper::paramsToUrl ( $params ) )->execute ()->getThalamusModel();
	}
	
	/**
	 * 
	 * @param integer $id
	 */
	public function rollbackImport($id) {
		return ThalamusRequest::delete( "/import/delete/$id" )->execute()->getRawResponse();
	}
	
	/**
	 * 
	 * @param integer $id
	 */
	public function cancelImport($id) {
		return ThalamusRequest::post( "/import/cancel/$id" )->execute()->getRawResponse();
	}
	
	/**
	 * 
	 * @param integer $id
	 */
	public function getImportStatus($id) {
		return ThalamusRequest::get( "/import/status/$id" )->execute()->getThalamusModel();
	}
	
	/**
	 * 
	 * @param string $email
	 * @param bool $active
	 */
	public function createEmail($email, $active = true) {
		
		$params = array(
			'email' => $email,
			'active' => $active
		);
		
		return ThalamusRequest::post( '/partner/emails', $params )->execute();
	}
	
	/**
	 * 
	 * @param string $email
	 * @param bool $active
	 */
	public function updateEmail($emailCurrent, $emailNew, $active = true) {
		
		$params = array(
				'email' => $emailNew,
				'active' => $active
		);
	
		return ThalamusRequest::put( "/partner/emails?email=$emailCurrent", $params )->execute();
	}
	
	/**
	 * 
	 */
	public function getEmails() {
		return ThalamusRequest::get( '/partner/emails' )->execute()->getThalamusModelList( 'ThalamusSDK\ThalamusModel', 'partnerEmails');
	} 
	
	public function getPrincipal() {
		return $this->session->getUser();
	}
	
	/**
	 * 
	 * @param string $host
	 * @param string $user
	 * @param string $password
	 * @param string $workingDirectory
	 * @param string $processorType
	 * @param string $dataSourceType
	 * @param string $touchpointCode
	 */
	public function createDataSource($host, $user, $password, $workingDirectory, $processorType, $dataSourceType, $touchpointCode) {
	
		$params = array(
			'host' => $host,
			'user' => $user,
			'password' => $password,
			'workingDirectory' => $workingDirectory,
			'processorType' => $processorType,
			'dataSourceType' => $dataSourceType,
			'touchpointCode' => $touchpointCode
		);
	
		return ThalamusRequest::post( '/partner/datasources', $params )->execute();
	}
	
	/**
	 * 
	 * @param integer $id
	 * @param string $host
	 * @param string $user
	 * @param string $password
	 * @param string $workingDirectory
	 * @param string $processorType
	 * @param string $dataSourceType
	 * @param string $touchpointCode
	 */
	public function updateDataSource($id, $host, $user, $password, $workingDirectory, $processorType, $dataSourceType, $touchpointCode) {
	
		$params = array(
				'host' => $host,
				'user' => $user,
				'password' => $password,
				'workingDirectory' => $workingDirectory,
				'processorType' => $processorType,
				'dataSourceType' => $dataSourceType,
				'touchpointCode' => $touchpointCode
		);
	
		return ThalamusRequest::put( "/partner/datasources/$id", $params )->execute();
	}
	
	/**
	 *
	 */
	public function getDataSources() {
		return ThalamusRequest::get( '/partner/datasources' )->execute()->getThalamusModelList( 'ThalamusSDK\ThalamusModel', 'datasources');
	}
	
	/**
	 * 
	 * @param integer $id
	 */
	public function getDataSource($id) {
		return ThalamusRequest::get( "/partner/datasources/$id"  )->execute()->getThalamusModel();
	}
	
	/**
	 *
	 * @param integer $id
	 */
	public function getTestDataSource($id) {
		return ThalamusRequest::get( "/partner/datasources/$id/test"  )->execute();
	}
	
	/**
	 * 
	 * @param unknown $principal
	 * @param string $password
	 * @param string $documentType
	 * @return unknown
	 */
	public function login($principal, $password = "", $oplogin = false) {

		$params = array (
			'principal' => $principal,
			'password'  => ! empty ( $password ) ? $password : self::DEFAULT_PASSWORD
		);

		if ($oplogin) {
			try {

				$r = ThalamusRequest::post ( '/signin', $params )->noAuth ()->execute();

			} catch ( \ThalamusSDK\ThalamusRequestException $e ) {

				$this->session->reset ();
				
				throw $e;

			}

			$party = $this->getPartyWithoutRunAs();

		} else {
			$this->session->reset ();
			
			$r = ThalamusRequest::post ( '/signin', $params )->noAuth ()->execute();

			$party = ThalamusRequest::get ( '/person' )->auth ( $principal, $password )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () );
		}

		$this->session->setUser ( $params ['principal'] );
		$this->session->setAccessToken ( $params ['password'] );
		$this->session->setPerson ( $party );
		
		return $party;
	}
	
	public function getParty($cache = true) {

			
			if ($this->session->getPerson() && $cache) {
				return $this->session->getPerson();
			} elseif ($this->session->getPerson()) {
			
				switch ($this->session->getPerson()->getProfile()->getPartyType()) {
					case 10:
						return $this->session->setPerson(ThalamusRequest::get ( '/person' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () ));
						break;
				}
			
			}

		return ThalamusRequest::get ( '/person' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () );
	}
	
	public function getPartyWithoutRunAs($cache = true) {
		
		if ($this->session->getPerson() && $cache) {
			return $this->session->getPerson();
		// } elseif ($this->session->getPerson()) {
		
		// 	switch ($this->session->getPerson()->getProfile()->getPartyType()) {
		// 		case 10:
		// 			return $this->session->setPerson(ThalamusRequest::get ( '/person' )->auth ( $this->session->getRealUser(), $this->session->getAccessToken() )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Person::className () ));
		// 			break;
		// 		case 11:
		// 			return $this->session->setPerson(ThalamusRequest::get ( '/company' )->auth ( $this->session->getRealUser(), $this->session->getAccessToken() )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Company::className () ));
		// 			break;
		
		// 			// 				case 14:
		// 			// 					ThalamusRequest::get ( '/phisicallocation' )->execute ()->getThalamusModel ( \ThalamusSDK\Model\Company::className () );
		// 			// 					break;
		// 	}
		}
	}
	
	public function getPartnersForPersonOperator() {
		return ThalamusRequest::get ( '/partner/partnersForPersonOperator' )->execute()->getThalamusModel();
	}
	
	public function setSudoId($id_party) {
		$this->session->setSudoId($id_party);
	}
	
	public function setSudoMode($bool) {
		$this->session->setSudo($bool);
	}

	public function getPartyId() {

		return $this->getParty()->getId();
	}

	public function loginProducer($login_id = null) {
		if ($login_id) {
			$user = $this->getPerson()->getProfile()->getEmail();
			$pass = $this->session->getAccessToken();

			return $this->login($user.RUNAS_SEPARATOR.$login_id, $pass, $oplogin = true);
		}
	}

	public function loginOperator() {
		$user = $this->getPerson()->getProfile()->getEmail();
		$pass = $this->session->getAccessToken();

		return $this->login(OPERATOR_PREFIX.$user, $pass, $oplogin = true);
	}

	public function loginP() {
		$user = $this->getPerson()->getProfile()->getEmail();
		$pass = $this->session->getAccessToken();

		return $this->login($user, $pass, $oplogin = false);
	}

	/**
	 * Verifica que el operador este logueado.
	 *
	 * @return void
	 */
	public function checkOperator() {
		if (strpos($this->getPrincipal(), OPERATOR_PREFIX) === false) {
			$this->loginOperator();
		}
	}

	/**
	 * Verifica que el productor este logueado.
	 *
	 * @param string $principal
	 *
	 * @return void
	 */
	public function checkLogin($principal = null)
	{
		if (strpos($this->getPrincipal(), $principal) === false) {
			$this->loginProducer($principal);
		}
	}
}
