<?php

namespace ThalamusSDK\Model;

use \ThalamusSDK\ThalamusModel;
use \ThalamusSDK\Model\Person\Profile;
use \ThalamusSDK\Model\Person\Consumer;
use \ThalamusSDK\Model\Person\OptIn;
use \ThalamusSDK\ThalamusHelper;
use \ThalamusSDK\ThalamusRequest;

/**
 * Class Person
 *
 * @package ThalamusSDK
 */
class Person extends ThalamusModel {
	
	/**
	 * return Party Id
	 */
	public function getId() {
		return $this->getProperty('id');
	}
	
	/**
	 */
	public function getConsumers() {
		return $this->getPropertyAsArray ( 'consumers', Consumer::classname () );
	}
	
	/**
	 */
	public function getOptIns() {
		return $this->getPropertyAsArray ( 'optIns', OptIn::classname () );
	}
	
	/**
	 *
	 * @param integer $channel        	
	 * @return Ambigous <NULL, OptIn>
	 */
	public function getOptIn($channel) {
		$optIn = null;
		
		foreach ( $this->getOptIns () as $item ) {
			if ($item->getChannelId () == $channel)
				$optIn = $item;
		}
		
		return $optIn;
	}
	
	/**
	 * get All activities
	 * 
	 * @return array of objects
	 */
	public function getActivities() {
		return $this->getActivity()->getAll();
	}
	
	/**
	 */
	public function getActivity() {
		$activity = $this->getProperty ( 'activities', Activity::classname () );
		
		if ($activity instanceof Activity) {
			return $activity;
		}
		
		throw new \ThalamusSDK\ThalamusOtherException ( 'ACTIVITIES_NOT_FOUND', 'ACTIVITIES_NOT_FOUND', 404 );
	}
	
	/**
	 * getEvent
	 *
	 * @param string $eventCode        	
	 * @return \ThalamusSDK\Model\Activity\Event\Event
	 */
	public function getEvent($eventCode) {
		return $this->getActivity ()->getEvent ( $eventCode );
	}
	
	/**
	 * getSurvey
	 *
	 * @param string $surveyCode        	
	 * @return \ThalamusSDK\Model\Activity\Survey\Survey
	 */
	public function getSurvey($surveyCode) {
		return $this->getActivity ()->getSurvey ( $eventCode );
	}
	
	/**
	 * getMileage
	 *
	 * @param string $mileageCode        	
	 * @return \ThalamusSDK\Model\Activity\Mileage\Mileage
	 */
	public function getMileage($mileageCode) {
		return $this->getActivity ()->getMileage ( $mileageCode );
	}
	
	/**
	 * getPayItForward
	 *
	 * @param string $payItForwardCode
	 * @return \ThalamusSDK\Model\Activity\PayItForward\PayItForward
	 */
	public function getPayItForward($payItForwardCode) {
		return $this->getActivity()->getPayItForward($payItForwardCode);
	}
	
	/**
	 */
	public function getProfile() {
		return $this->getProperty ( 'profile', Profile::classname () );
	}
	
	/**
	 *
	 * @param string $oldPassword        	
	 * @param string $newPassword        	
	 * @param string $confirmNewPassword        	
	 */
	public function changePassword($oldPassword, $newPassword, $confirmNewPassword) {
		$session = \ThalamusSDK\ThalamusSession::getSession();
		
		$params = array (
			"oldpassword" => $oldPassword,
			"newPassword" => $newPassword,
			"confirmNewPassword" => $confirmNewPassword 
		);
		
		$r = ThalamusRequest::put ( '/person/password', $params )->auth($session->getRealUser(), $session->getAccessToken())->execute ();
		
		$session->setAccessToken($newPassword);
	
		return $r;
	}
	
	/**
	 */
	public function getSocialConnections() {
		return ThalamusRequest::get ( '/connect', array () )->execute ()->getPropertyAsArray ( 'socialConnections', SocialConnection::classname () );
	}
	
	/**
	 * update updates profile/consumers/optIns data
	 *
	 * @param array $profile        	
	 * @param array $consumers        	
	 * @param array $optIns        	
	 *
	 * @return Person object
	 */
	public function update($profile = null, $optIns = null, $consumers = null) {
		$params = array ();
		
		if ($profile != null) {
			$params ['profile'] = $profile;
		}
		
		if ($consumers != null) {
			$params ['consumers'] = $consumers;
		} else {
			$params ['consumers'] = array ();
		}
		
		if ($optIns != null) {
			$params ['optIns'] = $optIns;
		}
		
		$response = ThalamusRequest::put ( '/person', $params )->execute ();
		
		$thalamus = \ThalamusSDK\Thalamus::getInstance ();
		
		$person = $thalamus->getPerson ( false );
		
		$session = \ThalamusSDK\ThalamusSession::getSession ();
		
		$session->setPerson ( $person );
		
		return $person;
	}
	
	/**
	 *
	 * @param boolean $read
	 * @param int $page
	 * @param int $size
	 *
	 * @return array of Message objects
	 */
	public function getInBox($read = null, $page = null, $size = null) {
		$params = array (
				'read' => $read,
				'page' => $page,
				'size' => $size 
		);
		
		return ThalamusRequest::get ( '/inbox/messages' . ThalamusHelper::paramsToUrl ( $params ) )->execute ()->getThalamusModelList ( Message::className (), 'messages' );
	}
	
	/**
	 *
	 * @return String Person Full Name
	 */
	public function __toString() {
		return $this->getProfile ()->getName ();
	}

	/**
	 *
	 * @return string Fiscal Name
	 */
	public function getFiscalName() {
		return ($this->getProperty( 'fiscalName' )) ? $this->getProperty( 'fiscalName' ) : $this->getProperty( 'name' );
	}

	/**
	 *
	 * @return int Cuit
	 */
	public function getCuit() {
		return ($this->getProperty( 'cuit' )) ? $this->getProperty( 'cuit' ) : $this->getProperty( 'fiscalId' );
	}

	/**
	 *
	 * @return string Province
	 */
	public function getProvince() {
		return $this->getProperty( 'address' ) ->getProperty( 'stateName' );
	}

	/**
	 *
	 * @return string Depto
	 */
	public function getDepto() {
		return $this->getProperty( 'address' ) ->getProperty( 'depto' );
	}

	/**
	 *
	 * @return string Location
	 */
	public function getLocation() {
		return $this->getProperty( 'address' ) ->getProperty( 'city' );
	}

	/**
	 *
	 * @return string categoria_elegidos
	 */
	public function getCategory() {
		return $this->getProperty( 'categoria_elegidos' );
	}
}