<?php
class RfIdUserType
{
	const OWNER				= 1;
	const NEIGHBOR			= 2;
	const VISITOR			= 3;
	const NEIGHBOR_EMPLOYEE	= 4;
	const SERVICE			= 5;
	const PROFESSIONAL		= 6;
	const WORKER			= 7;
	const GUARD				= 8;
	const GUEST				= 9;
	const PROVIDER			= 10;
	
	const FIRST				= 1;
	const LAST				= 10;
	
	public static function is_valid( $type )
	{
		return ( $type >= RfIdUserType::OWNER && $type <= RfIdUserType::PROVIDER );
	}
	
	public static function to_string( $type )
	{
		switch ( $type )
		{
			case RfIdUserType::OWNER: return "Propietario";
			case RfIdUserType::NEIGHBOR: return "Vecino";
			case RfIdUserType::VISITOR: return "Visita Frecuente";
			case RfIdUserType::NEIGHBOR_EMPLOYEE: return "Empleado del Barrio";
			case RfIdUserType::SERVICE: return "Servicio Auxiliar";
			case RfIdUserType::PROFESSIONAL: return "Profesional a cargo de obra";
			case RfIdUserType::WORKER: return "Empleado de Obra";
			case RfIdUserType::GUARD: return "Guardia";
			case RfIdUserType::GUEST: return "Visita Ocasional";
			case RfIdUserType::PROVIDER: return "Proveedor";
		}
	}
	
	public static function get_array()
	{
		$arr = array();
		
		for ( $i = RfIdUserType::FIRST; $i <= RfIdUserType::LAST; $i++ )
		{
			$arr[$i] = self::to_string( $i );
		}
		
		
		return $arr;
	}
}

class RfidUserStatus
{
	const ALLOWED	= 0;
	const SUSPENDED	= 1;
}

class RfidUserEntranceType
{
	const SIMPLE	= 0;
	const DOUBLE	= 1;
}

/**
	urfid_id serial NOT NULL,
	urfid_name character varying(64) NOT NULL DEFAULT ''::character varying,
	urfid_photo text,
	urfid_status smallint,
	urfid_restricted_access smallint,
	urfid_user_type smallint,
	urfid_user_data json,
	urfid_dni bigint,
	urfid_com_id bigint,
	urfid_plate character varying(64),
 **/
class RfidUser_model extends CI_Model
{
	public static $visitor_expiration_time = 15552000; // six months
	public static $guest_expiration_time = 2592000; // a month
	public static $fields = array( 'name', 'photo', 'status', 'restricted_access', 'user_type', 'user_data', 'dni', 'com_id', 'plate', 'phone', 'observations' );
	
	public static $basic_join = ' LEFT JOIN LATERAL 
								( SELECT string_agg(CAST( rup_plot AS text ), \',\') AS urfid_plot FROM rfid_user_plot WHERE rup_urfid_id = urfid_id ) AS plot ON TRUE ';
	
	public static function get_company_filter( $filter_company = NULL, $field_name = 'urfid_com_id' )
	{
		return SQL::get_or_filter( $filter_company, $field_name );
	}
	
	public static function get_company_filtered( $filter_company = NULL, $field_name = 'urfid_com_id' )
	{
		$filter = SQL::get_or_filter( $filter_company, $field_name );
		
		return '' != $filter ? ' AND ' . $filter : '';
	}
	
	public function add( $type, $com_id, $name, $dni, $plate, $phone, $observations, $plot = NULL )
	{
		$values = array( $name, '', RfidUserStatus::ALLOWED, 0, $type, "{}", $dni, $com_id, strtoupper( $plate ), $phone, $observations );
		
		$this->db->query( SQL::make_insert_pdo( 'rfid_user', self::$fields, '', 'urfid_' ), $values );
		
		$urfid_id = $this->db->insert_id();
		
		if ( RfIdUserType::OWNER == $type && isset( $plot ) && !empty( $plot ) )
		{
			$this->load->model('User_model');
			$this->load->model('Company_model');
			
			$data['uname'] = $this->Company_model->get_user_abbr( $com_id ) . str_replace( ',', '-', $plot );
			$data['upass'] = $dni;
			$data['umail'] = '';
			$data['name'] = '';
			$data['lastname'] = $name;
			$data['phone'] = $phone;
			$data['address'] = '';
			$data['support'] = '';
			$data['passkey'] = '';
			$data['observations'] = '';
			$data['ui_type'] = UserInfoType::USER_INFO_TYPE_DSHIELD;
			
			$uid = $this->User_model->create( $data, $com_id, $this->Company_model->get_city( $com_id ) );
			
			$this->update_uid( $urfid_id, $uid );
		}
		
		return $urfid_id;
	}
	
	public function update( $id, $name, $dni, $plate, $phone = '', $observations = '' )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'name', 'dni', 'plate', 'phone', 'observations', 'id' ), 'id', '', 'urfid_' ), array( $name, $dni, $plate, $phone, $observations, $id ) );
	}
	
	public function delete( $id )
	{
		$this->db->query( SQL::make_delete_pdo( 'rfid_user', 'urfid_id' ), array( $id ) );
	}
	
	public function get( $id )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user ' . self::$basic_join . ' WHERE urfid_id = ? LIMIT 1', ARRAY_A, array( $id ) );
	}
	
	public function get_full( $id )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user ' . self::$basic_join . ' INNER JOIN users ON urfid_uid = uid WHERE urfid_id = ? LIMIT 1', ARRAY_A, array( $id ) );
	}
	
	public function get_by_serial( $serial )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user ' . self::$basic_join . ' INNER JOIN rfid ON rfid_user = urfid_id 
									WHERE rfid_serial = ? LIMIT 1', ARRAY_A, array( $serial ) );
	}
	
	public function get_id_from_serial( $serial )
	{
		return $this->db->get_var( 'SELECT urfid_id FROM rfid_user INNER JOIN rfid ON rfid_user = urfid_id WHERE rfid_serial = ? LIMIT 1', array( $serial ) );
	}
	
	public function get_company( $urfid_id )
	{
		return $this->db->get_var( 'SELECT urfid_com_id FROM rfid_user WHERE urfid_id = ? LIMIT 1', array( $urfid_id ) );
	}
	
	public function get_by_dni( $dni, $filter_company )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user ' . self::$basic_join . ' WHERE urfid_dni = ? ' . self::get_company_filtered( $filter_company ) . ' LIMIT 1', ARRAY_A, array( $dni ) );
	}
	
	public function get_by_uid( $uid )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user ' . self::$basic_join . ' WHERE urfid_uid = ? LIMIT 1', ARRAY_A, array( $uid ) );
	}
	
	public function exists_uid( $uid )
	{
		return $this->db->get_var( 'SELECT 1 FROM rfid_user WHERE urfid_uid = ? LIMIT 1', array( $uid ) );
	}
	
	public function exists_dni( $dni, $filter_company )
	{
		return $this->db->get_var( 'SELECT 1 FROM rfid_user WHERE urfid_dni = ? ' . $this->get_company_filtered( $filter_company ) . ' LIMIT 1', array( $dni ) );
	}
	
	public function get_by_name( $name, $com_id )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user WHERE urfid_name = ? AND urfid_com_id = ? LIMIT 1', ARRAY_A, array( $name, $com_id ) );
	}
	
	public function get_by_plate( $plate, $filter_company, $get_first = FALSE )
	{
		$plate = SQL::escape_string( str_replace( ' ', '', $plate ) );
		
		$sql = "SELECT * FROM rfid_user " . self::$basic_join . " WHERE replace(urfid_plate, ' ', '') ILIKE '%{$plate}%' " . self::get_company_filtered( $filter_company ) . " ORDER BY urfid_user_type ASC";
		
		if ( $get_first )
		{
			return $this->db->get_row( $sql . ' LIMIT 1', ARRAY_A );
		}
		
		return $this->db->get_results( $sql, ARRAY_A );
	}
	
	public function exists_dni_not_me( $dni, $id, $filter_company )
	{
		return $this->db->get_var( 'SELECT 1 FROM rfid_user WHERE urfid_dni = ? AND urfid_id != ? ' . self::get_company_filtered( $filter_company ) . ' LIMIT 1', array( $dni, $id ) );
	}
	
	public function exists_uid_not_me( $uid, $id )
	{
		return $this->db->get_row( 'SELECT * FROM rfid_user WHERE urfid_uid = ? AND urfid_id != ? LIMIT 1', ARRAY_A, array( $uid, $id ) );
	}
	
	public function update_profile_photo( $id, $photo_name )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'photo', 'id' ), 'id', '', 'urfid_' ), array( $photo_name, $id ) );
	}
	
	public function update_data( $id, $data )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'user_data', 'id' ), 'id', '', 'urfid_' ), array( is_array( $data ) ? json_encode( $data ) : $data, $id ) );
	}
	
	public function update_status( $id, $status )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'status', 'id' ), 'id', '', 'urfid_' ), array( $status, $id ) );
	}
	
	public function update_expiration( $id, $expiration )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'expiration', 'id' ), 'id', '', 'urfid_' ), array( $expiration, $id ) );
	}
	
	public function update_uid( $id, $uid )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'uid', 'id' ), 'id', '', 'urfid_' ), array( $uid, $id ) );
	}
	
	public function update_uid_from_uid( $ouid, $uid )
	{
		$this->db->query( 'UPDATE rfid_user SET urfid_uid = ? WHERE urfid_uid = ?', array( $uid, $ouid ) );
	}
	
	public function update_user_type( $type, $id )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'user_type', 'id' ), 'id', '', 'urfid_' ), array( $type, $id ) );
	}
	
	public function update_entry_leave_status( $id, $val )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'inside', 'id' ), 'id', '', 'urfid_' ), array( $val, $id ) );
	}
	
	public function update_entrance_exit_type( $id, $entrance_type, $exit_type )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'entrance_type', 'exit_type', 'id' ), 'id', '', 'urfid_' ), array( $entrance_type, $exit_type, $id ) );
	}
	
	public function update_name( $id, $name )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'name', 'id' ), 'id', '', 'urfid_' ), array( $name, $id ) );
	}

	public function update_dni( $id, $dni )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'dni', 'id' ), 'id', '', 'urfid_' ), array( $dni, $id ) );
	}
	
	public function update_observations( $id, $observations )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'observations', 'id' ), 'id', '', 'urfid_' ), array( $observations, $id ) );
	}
	
	public function update_company_name( $id, $company_name )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_user', array( 'company_name', 'id' ), 'id', '', 'urfid_' ), array( $company_name, $id ) );
	}
	
	public function entry( $id )
	{
		$this->update_entry_leave_status( $id, 1 );
	}
	
	public function leave( $id )
	{
		$this->update_entry_leave_status( $id, 0 );
	}
	
	public function suspend( $id )
	{
		$this->update_status( $id, RfidUserStatus::SUSPENDED );
	}
	
	public function allow( $id )
	{
		$this->update_status( $id, RfidUserStatus::ALLOWED );
	}
	
	protected function build_inside_filter()
	{
		$fields = array(
			array(
				'field_name'	=>	'urfid_inside',
				'filter_val'	=>	1
			)/*,
			array(
				'field_name'	=>	'urfid_user_type',
				'filter_val'	=>	RfIdUserType::GUEST,
				'filter_type'	=>	SQLFilterType::UNEQUALS
			)*/
		);
		
		return SQL::build_query_filter( $fields );
	}
	
	public function count_inside( $filter_company )
	{
		return $this->count( $filter_company, $this->build_inside_filter() );
	}
	
	public function count( $filter_company, $filter = '' )
	{
		return $this->get_all( $filter_company, $filter, NULL, 1, 'COUNT(DISTINCT rfid_user.urfid_id)' );
	}
	
	public function get_all( $filter_company = NULL, $filter = NULL, $per_page = NULL, $page_num = 1, $fields_get = 'DISTINCT ON (rfid_user.urfid_name, rfid_user.urfid_id) rfid_user.*,rfid.*' )
	{
		$where		= self::get_company_filter( $filter_company );
		$is_count	= -1 != str_starts_with( 'COUNT', $fields_get );
		$plotit		= !$is_count ? ', plot.urfid_plot' : '';
		
		SQL::prepare_filter( $where, $filter, $is_count );
		
		$sql = 'SELECT ' . $fields_get . $plotit . '
				FROM rfid_user ' . self::$basic_join . '
				INNER JOIN company ON com_id = urfid_com_id 
				LEFT JOIN rfid ON rfid_user = urfid_id ' . $where;
		
		if ( NULL != $per_page )
		{
			$sql .= ' LIMIT '. (string)intval( $per_page ) .
					' OFFSET '.(string)intval( ($page_num-1)*$per_page );
		}
		
		if ( $is_count )
		{
			return $this->db->get_var( $sql );
		}
		else
		{
			return $this->db->get_results( $sql, ARRAY_A );
		}
	}
	
	public function owns_person( $com_id, $urfid_id )
	{
		return NULL != $this->db->get_var( 'SELECT 1 FROM rfid_user WHERE urfid_com_id = ? AND urfid_id = ? LIMIT 1', array( $com_id, $urfid_id ) );
	}
	
	public function owns_person_from_df( $com_df_id, $urfid_id )
	{
		return NULL != $this->db->get_var( 'SELECT 1 FROM rfid_user INNER JOIN company ON urfid_com_id = com_id WHERE com_df_id = ? AND urfid_id = ? LIMIT 1', array( $com_df_id, $urfid_id ) );
	}
	
	public function get_expired_insurances( $filter_company = NULL )
	{
		$sql = "SELECT * 
				FROM rfid_user 
				LEFT JOIN rfid_gallery ON rg_urfid_id = urfid_id AND rg_type = 2
				LEFT JOIN rfid_photo ON rp_gallery_id = rg_id
				WHERE 
					urfid_user_type = 7 AND 
					(
						(
							(urfid_user_data->'insurance_expiration') IS NOT NULL AND 
							(urfid_user_data->>'insurance_expiration')::bigint < EXTRACT(EPOCH FROM NOW()) 
						)
					 OR
						(urfid_user_data->'insurance_expiration') IS NULL
					)";
	}
	
	protected function get_not_validated_filter( $filter = NULL )
	{
		$fields = array(
			array(
				'field_name'	=>	'urfid_user_type',
				'filter_val'	=>	7
			),
			array(
				'field_name'	=>	'rp_state',
				'filter_val'	=>	1,
				'filter_type'	=>	SQLFilterType::UNEQUALS
			)
		);
		
		if ( isset( $filter ) )
		{
			$fields = array_merge( $fields, $filter );
		}
		
		return SQL::build_query_filter( $fields );
	}
	
	public function get_not_validated_insurances( $filter_company = NULL, $filter = NULL, $per_page = NULL, $page_num = 1, $fields_get = '*' )
	{
		$filter		= $this->get_not_validated_filter( $filter );
		$where		= self::get_company_filter( $filter_company );
		$is_count	= -1 != str_starts_with( 'COUNT', $fields_get );
		$plotit		= !$is_count ? ', plot.urfid_plot' : '';
		
		SQL::prepare_filter( $where, $filter, $is_count );
		
		$sql = 'SELECT ' . $fields_get . $plotit . ' 
				FROM rfid_user 
				' . self::$basic_join . '
				INNER JOIN rfid_gallery ON rg_urfid_id = urfid_id AND rg_type = 2
				INNER JOIN rfid_photo ON rp_gallery_id = rg_id ' . $where;
		
		if ( NULL != $per_page )
		{
			$sql .= ' LIMIT '. (string)intval( $per_page ) .
					' OFFSET '.(string)intval( ($page_num-1)*$per_page );
		}
		
		if ( $is_count )
		{
			return $this->db->get_var( $sql );
		}
		else
		{
			return $this->db->get_results( $sql, ARRAY_A );
		}
	}
	
	public function get_not_validated_insurances_count( $filter_company = NULL, $filter = NULL )
	{
		return $this->get_not_validated_insurances( $filter_company, NULL, NULL, 1, 'COUNT(*)' );
	}
	
	public function get_expired_guests_sql( $filter_company = NULL, $fields = 'urfid_id' )
	{
		$where		= self::get_company_filter( $filter_company );
		
		if ( '' != $where )
		{
			$where = ' AND ' . $where;
		}
		
		$sql = 'SELECT ' . $fields . ' FROM rfid_user 
				LEFT JOIN rfid_visit ON rv_urfid_id = urfid_id 
				WHERE 
					( rv_urfid_id IS NULL OR rv_time_end <= ' . SQL::unix_timestamp() . ' ) AND 
					urfid_user_type = ' . RfIdUserType::GUEST . ' AND 
					urfid_expiration >= ' . SQL::unix_timestamp() . $where;
		
		return $sql;
	}
	
	public function delete_expired_guests( $filter_company = NULL )
	{
		$sql = 'DELETE FROM rfid_user WHERE urfid_id IN ( ' . $this->get_expired_guests_sql() . ' )';
		
		$this->db->query( $sql );
	}
	
	public function set_expiration_for_guests()
	{
		$sql = 'UPDATE rfid_user SET urfid_expiration = ' . ( SQL::unix_timestamp() + RfidUser_model::$guest_expiration_time ) . ' 
				WHERE urfid_expiration IS NULL AND urfid_user_type = ' . RfIdUserType::GUEST;
		
		$this->db->query( $sql );
	}
	
	public function get_people_that_should_be_outside_sql( $fields = 'urfid_id', $get_last_entry_timestamp = FALSE, $where = '' )
	{
		$sql = "
					SELECT {$fields}" .
					( $get_last_entry_timestamp ? ', MAX(re_timestamp)' : '' )
				. "	FROM rfid_event 
					INNER JOIN rfid_user ON urfid_id = re_user
					WHERE 
						re_user = urfid_id AND 
						re_packet_type = 1 AND 
						re_event_type = 1 AND
						urfid_inside = 1 AND
						re_timestamp < (NOW())::abstime::int4 - 604800 
						{$where}
					GROUP BY urfid_id";
		
		return $sql;
	}
	
	public function get_people_that_should_be_outside( $fields = '*', $get_last_entry_timestamp = TRUE, $where = '' )
	{
		return $this->db->get_results( $this->get_people_that_should_be_outside_sql( $fields, $get_last_entry_timestamp, $where ) );
	}
	
	public function update_inside_status()
	{
		$outside_sql = $this->get_people_that_should_be_outside_sql();
		
		$update_sql = "UPDATE rfid_user SET urfid_inside = 0 WHERE urfid_id IN ( {$outside_sql} )";
		
		$this->db->query( $update_sql );
	}
	
	public function get_near_to_expire_workers_insurance( $filter_company = NULL, $fields_get = "rfid_user.*, urfid_user_data->>'insurance_expiration' AS insurance_expiration, urfid_professional_id, urfid_professional_name" )
	{
		$where			= self::get_company_filter( $filter_company );
		$company_where	= self::get_company_filter( $filter_company, 'urfid_prof.urfid_com_id' );
		
		if ( '' != $company_where )
		{
			$company_where = ' AND ' . $company_where;
		}
		
		$is_count		= -1 != str_starts_with( 'COUNT', $fields_get );
		$plotit			= !$is_count ? ', plot.urfid_plot' : '';
		$filter['filter'] = "(urfid_user_data->>'insurance_expiration')::bigint <= " . SQL::unix_timestamp() . '+864000';
		
		SQL::prepare_filter( $where, $filter, $is_count );
		
		$sql = 'SELECT ' . $fields_get . $plotit . ' FROM rfid_user ' . 
				self::$basic_join . " 
				INNER JOIN rfid_user_plot ON urfid_id = rup_urfid_id  
				LEFT JOIN LATERAL
					(	SELECT string_agg(CAST( urfid_id AS text ), ',') AS urfid_professional_id, string_agg(urfid_name, ',') AS urfid_professional_name 
						FROM rfid_user_plot AS rup_lat 
						INNER JOIN rfid_user AS urfid_prof ON urfid_prof.urfid_id = rup_lat.rup_urfid_id 
						WHERE rup_lat.rup_plot = rfid_user_plot.rup_plot AND urfid_prof.urfid_user_type = " . RfIdUserType::PROFESSIONAL . ' ' . $company_where . ' 
					) AS professional ON TRUE ' . $where;
		
		if ( $is_count )
		{
			return $this->db->get_var( $sql );
		}
		else
		{
			return $this->db->get_results( $sql, ARRAY_A );
		}
	}
	
	public function get_near_to_expire_workers_insurance_count( $filter_company = NULL )
	{
		return $this->get_near_to_expire_workers_insurance( $filter_company, "COUNT(DISTINCT( urfid_id, urfid_name ))" );
	}
	
	public function get_recently_created_workers( $filter_company = NULL, $fields_get = '*', $time = 432000 )
	{
		$where		= self::get_company_filter( $filter_company );
		$is_count		= -1 != str_starts_with( 'COUNT', $fields_get );
		$plotit			= !$is_count ? ', plot.urfid_plot' : '';
		$filter['filter'] = 'urfid_creation_time >= ' . SQL::unix_timestamp() . '-' . (string)$time . ' AND urfid_user_type = ' . RfIdUserType::WORKER;
		
		SQL::prepare_filter( $where, $filter, $is_count );
		
		$sql = 'SELECT ' . $fields_get . $plotit . ' FROM rfid_user ' . $where;
		
		if ( $is_count )
		{
			return $this->db->get_var( $sql );
		}
		else
		{
			return $this->db->get_results( $sql, ARRAY_A );
		}
	}
	
	public function get_recently_created_workers_count( $filter_company = NULL, $time = 432000 )
	{
		return $this->get_recently_created_workers( $filter_company, 'COUNT(*)', $time );
	}
}
