<?php

// state = el estado indica que tipo de acceso fué     0x00 = Sin identificar, 0x01  = ingreso, 0x02= egreso, 0x03 = acceso denegado (para este casolo tenes que guardar para asignarlo como activo)
class RfidEventType
{
	const UNKNOWN = 0x00;
	const ENTRY = 0x01;
	const LEAVE = 0x02;
	const DENIED_ACCESS = 0x03;
	const UNKNOWN_VIP = 0x04;
	const ENTRY_VIP = 0x05;
	const LEAVE_VIP = 0x06;
}

//type_device, tipo de dispositivo  0x01 = RFID  0x02 = control remoto
class RfidType
{
	const RFID = 0x01;
	const REMOTE_CONTROL = 0x02;
	const OPEN = 0x05;
	const UPDATE_CITYMESH = 0x79;
	const GENERAL_PHOTO = 0x7A;
	const TOOL_PHOTO = 0x7B;
	const INSURANCE_PHOTO = 0x7C;
	const TIME_FRAME = 0x7D;
}

class RfidResponseType
{
	const EXECUTE = 0x01;
	const STATE = 0x80;
	const ERASE = 0x81;
	const ADD = 0x82;
	const OPEN = 0x85;
}

class RfidCommandType
{
	const ERASE = 0x00;
	const ADD = 0x01;
	const QUERY = 0x02;
	const ADD_VIP = 0x03;
	const ANY = 0xEE;
}

class RfidResponseStateType
{
	const RESP_OK = 0x00;
	const RESP_ERROR = 0xFA;
	const RESP_INVALID_CMD = 0xFB;
	const RESP_INVALID_VAL = 0xFC;
	const RESP_TIMEOUT = 0xFD;
}

class RfidEventResultType
{
	const NONE					= 0;
	const ACCESS_OUT_OF_TIME	= 1;
	const EXPIRED_INSURANCE		= 2;
	const EXPIRING_INSURANCE	= 4;
	const LOGGED_WITH_DNI		= 8;
	const ACCESS_WALKING		= 16;
	const INVALID_INSURANCE		= 32;
	const NO_INSURANCE			= 64;
	const EXPIRING_VISIT		= 128;
	const EXPIRED_VISIT			= 256;
	const SUSPENDED_RFID		= 512;
	const USER_TYPE_NOT_ALLOWED = 1024;
	const GUEST_VISIT			= 2048;
	
	public static function to_string( $type )
	{
		$res = "";
		
		if ( $type & RfidEventResultType::ACCESS_OUT_OF_TIME )
		{
			$res .= " Acceso fuera de horario permitido.";
		}
		
		if ( $type & RfidEventResultType::EXPIRED_INSURANCE )
		{
			$res .= " Acceso con seguro de trabajo expirado.";
		}
		
		if ( $type & RfidEventResultType::EXPIRING_INSURANCE )
		{
			$res .= " Acceso con seguro de trabajo por expirar.";
		}
		
		if ( $type & RfidEventResultType::NO_INSURANCE )
		{
			$res .= " Acceso sin seguro de trabajo.";
		}
		
		if ( $type & RfidEventResultType::INVALID_INSURANCE )
		{
			$res .= " Acceso con seguro de trabajo sin validar.";
		}
		
		if ( $type & RfidEventResultType::EXPIRING_VISIT )
		{
			$res .= " Visita por expirar.";
		}
		
		if ( $type & RfidEventResultType::EXPIRED_VISIT )
		{
			$res .= " Visita expirada.";
		}
		
		if ( $type & RfidEventResultType::SUSPENDED_RFID )
		{
			$res .= " RFID suspendida.";
		}
		
		if ( $type & RfidEventResultType::USER_TYPE_NOT_ALLOWED )
		{
			$res .= " Tipo de usuario no permitido en acceso.";
		}
		
		if ( $type & RfidEventResultType::GUEST_VISIT ) {
			$res .= " Visita ocasional.";
		}
		
		return $res;
	}
}


/** TABLE REFERENCE
	re_id serial NOT NULL,
	re_serial bigint,
	re_code bigint,
	re_packet_type smallint,
	re_event_type smallint,
	re_timestamp bigint,
	re_state character varying( 18),
 **/
class RfidEvent_model extends CI_Model
{
	public static $fields = array( 'serial', 'code', 'packet_type', 'event_type', 'timestamp', 'com_id', 'state', 'user', 'result', 'user_data', 'user_name', 'user_dni', 'user_plate', 'user_plot', 'user_type', 'note' );
	
	public static $basic_join = ' LEFT JOIN LATERAL 
									( SELECT string_agg(CAST( rup_plot AS text ), \',\') AS urfid_plot FROM rfid_user_plot WHERE rup_urfid_id = urfid_id ) AS plot ON TRUE ';
	
	public function add( $serial, $code, $packet_type, $event_type, $timestamp, $com_id, $state = 'OK', $user_id = NULL, $result = RfidEventResultType::NONE )
	{
		$user_data = NULL;
		$user_name = NULL;
		$user_dni = NULL;
		$user_plate = NULL;
		$user_plot = NULL;
		$user_type = NULL;
		$note = NULL;
		
		if ( NULL != $user_id )
		{
			$this->load->model('RfidUser_model');
			
			$user = $this->RfidUser_model->get( $user_id );
			
			if ( isset( $user ) )
			{
				$user_data = json_encode( $user );
				$user_name = $user['urfid_name'];
				$user_dni = $user['urfid_dni'];
				$user_plate = $user['urfid_plate'];
				$user_plot = isset( $user['urfid_plot'] ) ? '{' . $user['urfid_plot'] . '}' : NULL;
				$user_type = $user['urfid_user_type'];
			}
		}
		
		$this->db->set_qd( array( $serial, $code, $packet_type, $event_type, $timestamp, $com_id, $state, $user_id, $result, $user_data, $user_name, $user_dni, $user_plate, $user_plot, $user_type, $note ) );
		
		$this->db->query( SQL::make_insert_pdo( 'rfid_event', self::$fields, '', 're_' ) );
		
		return $this->db->insert_id();
	}
	
	// Add command to send to all citymeshs registered in the system
	public function add_all( $serial, $packet_type, $event_type, $timestamp, $com_id, $state = 'OK', $user_id = NULL, $result = RfidEventResultType::NONE )
	{
		$rcities = $this->db->get_results( 'SELECT * FROM rfid_citymesh WHERE rc_com_id = ?', ARRAY_A, array( $com_id ) );
		
		if ( isset( $rcities ) )
		{
			foreach ( $rcities as $rcity )
			{
				$this->add( $serial, $rcity['rc_code'], $packet_type, $event_type, $timestamp, $com_id, $state, $user_id, $result );
			}
		}
	}
	
	public function update_user_data( $re_id, $user_id )
	{
		$user_data = NULL;
		$user_name = NULL;
		$user_dni = NULL;
		$user_plate = NULL;
		$user_plot = NULL;
		$user_type = NULL;
		
		if ( NULL != $user_id )
		{
			$this->load->model('RfidUser_model');
			
			$user = $this->RfidUser_model->get( $user_id );
			
			if ( isset( $user ) )
			{
				$user_data = json_encode( $user );
				$user_name = $user['urfid_name'];
				$user_dni = $user['urfid_dni'];
				$user_plate = $user['urfid_plate'];
				$user_plot = isset( $user['urfid_plot'] ) ? '{' . $user['urfid_plot'] . '}' : NULL;
				$user_type = $user['urfid_user_type'];
			}
		}
		
		$fields = array( 'user_data', 'user_name', 'user_dni', 'user_plate', 'user_plot', 'user_type', 'id' );
		$values = array( $user_data, $user_name, $user_dni, $user_plate, $user_plot, $user_type, $re_id );
		
		$this->db->query( SQL::make_update_pdo( 'rfid_event', $fields, 'id', '', 're_' ), $values );
	}
	
	public function get_event_desc( $event_type, $serial, $code, $timestamp, $packet_type = RfidType::RFID, $state = '', $result = RfidEventResultType::NONE )
	{
		switch ( $packet_type )
		{
			case RfidType::RFID:
			{
				if ( 'WAITING_RESPONSE' == $state || 'PENDING_VALIDATE' == $state || 'ACCEPTED' == $state || 'REJECTED' == $state || 'ERROR_TIMEOUT' == $state )
				{
					switch ( $event_type )
					{
						case RfidEventType::UNKNOWN:
						case RfidEventType::ENTRY:
						case RfidEventType::LEAVE:
						{
							$type_access = 'Cargadora de tarjetas';
							
							if ( $event_type == RfidEventType::ENTRY )
							{
								$type_access = 'Ingreso';
							}
							else if ( $event_type == RfidEventType::LEAVE )
							{
								$type_access = 'Salida';
							}
							
							if ( $result & RfidEventResultType::ACCESS_WALKING )
							{
								$type_access .= ' peatonal';
							}
							
							$log_type = 'RFID';
							
							if ( $result & RfidEventResultType::LOGGED_WITH_DNI )
							{
								$log_type = 'DNI: ' . $serial;
							}
							else
							{
								$log_type.= ': MAC ' . dechex( $serial );
							}
							
							$msg = $type_access . ' ' . $log_type;
							
							if ( 0 != $code )
							{
								$msg.= ' Número: ' . dechex( $code );
							}
							
							if ( 'ERROR_TIMEOUT' == $state )
							{
								$msg.= '. Sin respuesta.';
							}
							
							return $msg;
						}
						case RfidEventType::DENIED_ACCESS:
						{
							return 'Acceso denegado para RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
						case RfidEventType::UNKNOWN_VIP:
						{
							return 'Acceso VIP RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
						case RfidEventType::ENTRY_VIP:
						{
							return 'Ingreso VIP de RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
						case RfidEventType::LEAVE_VIP:
						{
							return 'Egreso VIP de RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
					}
				}
				else
				{
					switch ( $event_type )
					{
						case RfidCommandType::ERASE:
						{
							return 'Removida tarjeta RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
						case RfidCommandType::ADD:
						{
							return 'Añadida tarjeta RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
						}
					}
				}
			}
			case RfidType::REMOTE_CONTROL:
			{
				break;
			}
			case RfidType::OPEN:
			{
				if ( $result & RfidEventResultType::LOGGED_WITH_DNI )
				{
					return 'Apertura de barrera para DNI ' . $serial . ' Número: ' . dechex( $code );
				}
				
				return 'Apertura de barrera para RFID: MAC ' . dechex( $serial ) . ' Número: ' . dechex( $code );
			}
			case RfidType::UPDATE_CITYMESH:
			{
				load_model('RfidCitymesh_model');
				
				$city_type = RfidCitymeshType::is_entry( RfidCitymeshType::from_event_type( $event_type ) ) ? 'ingreso' : 'egreso';
				
				return 'Citymesh número : ' . dechex( $code ) . ' actualizado como puerta de ' . $city_type;
			}
			case RfidType::TOOL_PHOTO:
			{
				$RfidPhoto_model	= load_model('RfidPhoto_model');
				$RfidUser_model	= load_model('RfidUser_model');
				
				$photo = $RfidPhoto_model->get_unlinked( $serial );
				$user = $RfidUser_model->get( $code );
				
				switch ( $event_type )
				{
					case RfidCommandType::ERASE:
					{
						return '<a href="' . base_url('/assets/photos/'.$photo['rp_path'].'.jpg') . '" target="_blank">Herramienta</a> removida del usuario: ' . $user['urfid_name'];
					}
					case RfidCommandType::ADD:
					{
						return '<a href="' . base_url('/assets/photos/'.$photo['rp_path'].'.jpg') . '" target="_blank">Herramienta</a> añadida al usuario: ' . $user['urfid_name'];
					}
				}
			}
			case RfidType::TIME_FRAME:
			{
				$CompanyTimeFrame_model = load_model('CompanyTimeFrame_model');
				
				$ctf_event	= $CompanyTimeFrame_model->get_id( $serial );
				$duration	= ( $ctf_event['ctf_time_end'] - $ctf_event['ctf_time_start'] ) / 60;
				$type		= $event_type == 0 ? 'ingreso' : 'egreso';
				
				return 'Ventana de tiempo para puerta de ' . $type . ' activada por ' . $duration . ' minutos.';
			}
		}
		
		return '';
	}
	
	public function get_event_desc_full( $event_type, $serial, $code, $timestamp, $packet_type = RfidType::RFID, $state = '', $result = 0, $validator = NULL )
	{
		$event_desc = $this->get_event_desc( $event_type, $serial, $code, $timestamp, $packet_type, $state, $result );
		
		if ( $packet_type == RfidType::RFID )
		{
			switch ( $event_type )
			{
				case RfidEventType::ENTRY:
				case RfidEventType::LEAVE:
				{
					if ( 'ACCEPTED' == $state || 'REJECTED' == $state )
					{
						$event_desc.= $state == 'ACCEPTED' ? '. Acceso ACEPTADO' : '. Acceso RECHAZADO';
						
						if ( isset( $validator ) )
						{
							$event_desc.= " por " . $validator;
						}
						
						$event_desc .= '.';
					}
					else if ( 'PENDING_VALIDATE' == $state )
					{
						$event_desc.= 'Acceso Pendiente de Validación';
					}
					
					break;
				}
			}
		}
		
		$event_desc .= RfidEventResultType::to_string( $result );
		
		return $event_desc;
	}
	
	public function get_last_id()
	{
		return $this->db->get_var( "SELECT re_id FROM rfid_event ORDER BY re_id DESC LIMIT 1" );
	}
	
	public function get_last_command_id( $command_type, $serial, $code )
	{
		return $this->db->get_var( "SELECT re_id FROM rfid_event WHERE 
																re_packet_type = ? AND 
																re_serial = ? AND 
																re_code = ? AND 
																re_state = 'WAITING_RESPONSE'
																ORDER BY re_id DESC
																LIMIT 1", array( $command_type, $serial, $code ) );
	}
	
	public function update_state( $id, $state )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_event', array( 'state', 'id' ), 'id', '', 're_' ), array( $state, $id ) );
	}
	
	public function update_note( $id, $note )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_event', array( 'note', 'id' ), 'id', '', 're_' ), array( $note, $id ) );
	}
	
	public function update_validator( $id, $validator )
	{
		$this->db->query( SQL::make_update_pdo( 'rfid_event', array( 'validator', 'id' ), 'id', '', 're_' ), array( $validator, $id ) );
	}
	
	public function update_command_state( $command_type, $serial, $code, $event_type, $state_id )
	{
		switch ( $command_type )
		{
			case RfidResponseType::ADD:
			case RfidResponseType::ERASE:
			case RfidResponseType::STATE:
			{
				$command_type = RfidType::RFID;
				break;
			}
			case RfidResponseType::OPEN:
			{
				$command_type = RfidType::OPEN;
				break;
			}
		}
		
		$re_id = $this->get_last_command_id( $command_type, $serial, $code );
		
		if ( isset( $re_id ) )
		{
			switch ( $state_id )
			{
				case RfidResponseStateType::RESP_OK:
				{
					$this->update_state( $re_id, 'EXECUTED' ); break;
				}
				case RfidResponseStateType::RESP_INVALID_CMD:
				{
					$this->update_state( $re_id, 'ERROR_INVALID_CMD' ); break;
				}
				case RfidResponseStateType::RESP_INVALID_VAL:
				{
					$this->update_state( $re_id, 'ERROR_INVALID_VAL' ); break;
				}
				case RfidResponseStateType::RESP_TIMEOUT:
				{
					$this->update_state( $re_id, 'ERROR_TIMEOUT' ); break;
				}
				case RfidResponseStateType::RESP_ERROR:
				{
					$this->update_state( $re_id, 'ERROR' ); break;
				}
			}
		}
	}
	
	public function get_pending_commands( $com_id )
	{
		$sql = "SELECT * FROM rfid_event WHERE re_state = 'PENDING_SEND' AND re_com_id = ? ORDER BY re_id ASC";
		
		return $this->db->get_results( $sql, ARRAY_A, array( $com_id ) );
	}
	
	public function set_waiting_response( $com_id )
	{
		$this->db->query( "UPDATE rfid_event SET re_state = 'WAITING_RESPONSE' WHERE re_state = 'PENDING_SEND' AND re_com_id = ?", array( $com_id ) );
	}
	
	public function set_waiting_response_by_event_id( $re_id )
	{
		$this->db->query( "UPDATE rfid_event SET re_state = 'WAITING_RESPONSE' WHERE re_state = 'PENDING_SEND' AND re_id = ?", array( $re_id ) );
	}
	
	public function get_pending_events( $uid, $com_id, $filter = NULL )
	{
		$where		= "ruc_uid = ? AND re_com_id = ? AND re_state = 'PENDING_VALIDATE'";
		
		SQL::prepare_filter( $where, $filter, FALSE );
		
		$sql = "SELECT * 
				FROM rfid_event 
				INNER JOIN rfid_citymesh ON rc_code = re_code 
				INNER JOIN rfid_user_citymesh ON ruc_citymesh_id = rc_id 
				INNER JOIN rfid ON rfid_serial = re_serial
				LEFT JOIN rfid_user ON urfid_id = rfid_user
				$where";
		
		return $this->db->get_results( $sql, ARRAY_A, array( $uid, $com_id ) );
	}
	
	public function get_rejected_events( $com_id, $filter = NULL )
	{
		$where		= "re_com_id = ? AND re_state = 'REJECTED'";
		
		SQL::prepare_filter( $where, $filter, FALSE );
		
		$sql = "SELECT * 
				FROM rfid_event 
				INNER JOIN rfid_citymesh ON rc_code = re_code 
				INNER JOIN rfid ON rfid_serial = re_serial
				LEFT JOIN rfid_user ON urfid_id = rfid_user
				$where";
		
		return $this->db->get_results( $sql, ARRAY_A, array( $com_id ) );
	}
	
	public function get_event( $re_id )
	{
		$sql = "SELECT * 
				FROM rfid_event 
				INNER JOIN rfid_citymesh ON rc_code = re_code 
				INNER JOIN rfid_user_citymesh ON ruc_citymesh_id = rc_id 
				INNER JOIN rfid ON rfid_serial = re_serial
				LEFT JOIN rfid_user ON urfid_id = rfid_user
				WHERE re_id = ?
				LIMIT 1";
		
		return $this->db->get_row( $sql, ARRAY_A, array( $re_id ) );
	}
	
	public function get_last_urfid_event( $urfid_id )
	{
		$sql = "SELECT * 
				FROM rfid_event 
				INNER JOIN rfid_citymesh ON rc_code = re_code 
				INNER JOIN rfid ON rfid_serial = re_serial
				INNER JOIN rfid_user ON urfid_id = rfid_user
				WHERE re_user = ?
				ORDER BY re_timestamp DESC
				LIMIT 1";
		
		return $this->db->get_row( $sql, ARRAY_A, array( $urfid_id ) );
	}
	
	public function get_last_id_entry_event( $urfid_id, $access_type, $from_timestamp )
	{
		$sql = "SELECT * 
				FROM rfid_event 
				WHERE re_user = ? AND re_timestamp >= ? AND re_event_type = ?
				ORDER BY re_timestamp DESC
				LIMIT 1";
		
		return $this->db->get_row( $sql, ARRAY_A, array( $urfid_id, $from_timestamp, $access_type ) );
	}
	
	public function get_last_dni_entry_event( $dni, $access_type, $from_timestamp )
	{
		$sql = "SELECT * 
				FROM rfid_event 
				WHERE re_serial = ? AND re_timestamp >= ? AND re_event_type = ?
				ORDER BY re_timestamp DESC
				LIMIT 1";
		
		return $this->db->get_row( $sql, ARRAY_A, array( $dni, $from_timestamp, $access_type ) );
	}
	
	public function set_serial_accept_on_pending_validation( $serial, $result = RfidEventResultType::NONE )
	{
		$this->db->query( "UPDATE rfid_event SET re_state = 'ACCEPTED', re_result = ? WHERE re_state = 'PENDING_VALIDATE' AND re_serial = ?", array( $result, $serial ) );
	}
	
	public function set_serial_reject_on_pending_validation( $serial, $result )
	{
		$this->db->query( "UPDATE rfid_event SET re_state = 'REJECTED', re_result = ? WHERE re_state = 'PENDING_VALIDATE' AND re_serial = ?", array( $result, $serial ) );
	}
	
	public static function get_company_filter( $filter_company = NULL, $field_name = 're_com_id' )
	{
		return SQL::get_or_filter( $filter_company, $field_name );
	}
	
	public function count( $filter_company, $filter = '' )
	{
		return $this->get_all( $filter_company, $filter, NULL, 1, 'COUNT(*)' );
	}
	
	public function get_all( $filter_company = NULL, $filter = NULL, $per_page = NULL, $page_num = 1, $fields_get = '*' )
	{
		$where		= self::get_company_filter( $filter_company );
		$is_count	= -1 != str_starts_with( 'COUNT', $fields_get );
		
		SQL::prepare_filter( $where, $filter, $is_count );
		
		$sql = 'SELECT ' . $fields_get . '
				FROM rfid_event 
				LEFT JOIN rfid_user ON urfid_id = re_user ' . self::$basic_join . $where;
		
		if ( NULL != $per_page )
		{
			$sql .= ' LIMIT '. (string)intval( $per_page ) .
					' OFFSET '.(string)intval( ($page_num-1)*$per_page );
		}
		
		if ( $is_count )
		{
			return $this->db->get_var( $sql );
		}
		else
		{
			return $this->db->get_results( $sql, ARRAY_A );
		}
	}
	
	public function get_event_result( $urfid_id, $rc_id, $person = NULL )
	{
		$data['person'] = $person = ( NULL == $person ) ? load_model('RfidUser_model')->get( $urfid_id ) : $person;
		
		$data['user_data'] = $user_data = json_decode( $data['person']['urfid_user_data'], TRUE );
		
		$data['access_allowed'] = load_model('RfidUserHours_model')->access_allowed( $urfid_id );
		
		$data['re_result'] = 0;
		
		if ( !$data['access_allowed'] )
		{
			$data['re_result'] |= RfidEventResultType::ACCESS_OUT_OF_TIME;
		}
		
		$insurance = load_model('RfidPhoto_model')->get_insurance_from( $urfid_id );
		
		if ( isset( $insurance ) )
		{
			if ( $insurance['rp_state'] != InsurancePhotoState::VALID )
			{
				$data['re_result'] |= RfidEventResultType::INVALID_INSURANCE;
				$data['invalid_insurance'] = TRUE;
			}
		}
		else if ( $person['urfid_user_type'] == RfIdUserType::WORKER )
		{
			$data['re_result'] |= RfidEventResultType::NO_INSURANCE;
			$data['no_insurance'] = TRUE;
		}
		
		$day_secs = 60*60*24;
		
		if ( isset( $user_data['insurance_expiration'] ) )
		{
			$exp_time = $user_data['insurance_expiration'] + $day_secs - 1; // check the complete day
			
			if ( time() > $exp_time )
			{
				$data['expired_insurance'] = TRUE;
				$data['re_result'] |= RfidEventResultType::EXPIRED_INSURANCE;
			}
			else if ( time() > $user_data['insurance_expiration'] - ( $day_secs * 10 ) )
			{
				$data['expiring_insurance'] = TRUE;
				$data['re_result'] |= RfidEventResultType::EXPIRING_INSURANCE;
				
				if ( $user_data['insurance_expiration'] - time() <= 0 )
				{
					$data['expiring_time'] = "EL SEGURO DE TRABAJO EXPIRA HOY";
				}
				else
				{
					$days_to_expire = (int)( ( $user_data['insurance_expiration'] - time() ) / $day_secs );
					$data['expiring_time'] = "EL SEGURO DE TRABAJO VENCE EN " . $days_to_expire . " DÍAS";
				}
			}
		}
		
		if ( $person['urfid_user_type'] == RfIdUserType::GUEST )
		{
			$RfidVisit_model = load_model('RfidVisit_model');
			
			if ( !$RfidVisit_model->allowed( $urfid_id ) )
			{
				$data['is_guest'] = TRUE;
			}
			
			$data['re_result'] |= RfidEventResultType::GUEST_VISIT;
		}
		else if ( $person['urfid_user_type'] == RfIdUserType::VISITOR && isset( $person['urfid_expiration'] ) )
		{
			$exp_time = $person['urfid_expiration'] + $day_secs - 1; // check the complete day
			
			if ( time() > $exp_time )
			{
				$data['expired_visit'] = TRUE;
				$data['re_result'] |= RfidEventResultType::EXPIRED_VISIT;
				
				if ( $person['urfid_status'] != RfidUserStatus::SUSPENDED )
				{
					$person['urfid_status'] = RfidUserStatus::SUSPENDED;
					
					$RfidUser_model = load_model('RfidUser_model');
					
					$RfidUser_model->update_status( $person['urfid_id'], $person['urfid_status'] );
				}
			}
			else if ( time() > $person['urfid_expiration'] - ( $day_secs * 10 ) )
			{
				$data['expiring_visit'] = TRUE;
				$data['re_result'] |= RfidEventResultType::EXPIRING_VISIT;
				
				if ( $person['urfid_expiration'] - time() <= 0 )
				{
					$data['expiring_time'] = "TARJETA DE VISITA VENCE HOY";
				}
				else
				{
					$days_to_expire = (int)( ( $person['urfid_expiration'] - time() ) / $day_secs );
					$data['expiring_time'] = "VENCIMIENTO DE TARJETA DE VISITA EN " . $days_to_expire . " DÍAS";
				}
			}
		}
		
		if ( $person['urfid_status'] == RfidUserStatus::SUSPENDED )
		{
			$data['suspended_rfid'] = TRUE;
			$data['re_result'] |= RfidEventResultType::SUSPENDED_RFID;
		}
		
		if ( NULL != $rc_id )
		{
			if ( !( load_model('RfidCitymeshRestriction_model')->can_access( $person['urfid_com_id'], $person['urfid_user_type'], $rc_id ) ) )
			{
				$citymesh = load_model('RfidCitymesh_model')->get_id( $rc_id );
				
				$data['user_type_not_allowed'] = TRUE;
				
				$data['user_type_error'] = "El usuario es de tipo " . RfIdUserType::to_string( $person['urfid_user_type'] ) . ", y los usuarios de este tipo no pueden ingresar por el acceso: " . $citymesh['rc_name'];
				
				$data['re_result'] |= RfidEventResultType::USER_TYPE_NOT_ALLOWED;
			}
		}
		
		return $data;
	}
	
	protected function gen_graph_subquery( $type, $start_time, $end_time, $filter, $priority = 0 )
	{
		$fixed_type	= str_replace( ' ', '_', $type );
		
		$series_sql = "( SELECT '{$type}' AS type, generate_series AS time, 0 AS num FROM generate_series( {$start_time} , {$end_time}, 3600 ) ) AS base_serie";
		
		$real_sql 	= "( SELECT '{$type}' AS type, ( floor( re_timestamp / 3600 ) * 3600 ) AS time, COUNT(re_event_type) AS num
						FROM rfid_event
						WHERE re_timestamp >= {$start_time} AND re_timestamp <= {$end_time} AND {$filter}
						GROUP BY ( floor( re_timestamp / 3600 ) * 3600 ) ) AS {$fixed_type}";
						
		$full_sql	=  "SELECT base_serie.type, base_serie.time, COALESCE({$fixed_type}.num,0) AS total, {$priority} AS priority
						FROM {$series_sql}
						LEFT JOIN {$real_sql} ON base_serie.time = {$fixed_type}.time";
		
		return $full_sql;
	}
	
	protected function gen_graph_subquery_hw_code( $type, $start_time, $end_time, $filter, $rc_type, $priority = 0 )
	{
		$fixed_type	= str_replace( ' ', '_', $type );
		
		$series_sql = "( SELECT rc_code, '{$type} ' || to_hex(rc_code)::text AS type, generate_series AS time, 0 AS num FROM generate_series( {$start_time} , {$end_time}, 3600 )
						INNER JOIN rfid_citymesh ON TRUE WHERE rc_type = {$rc_type} GROUP BY rc_code, generate_series ORDER BY time ASC ) AS base_serie";
		
		$real_sql =  "( SELECT re_code AS rc_code, '{$type} ' || to_hex(re_code)::text AS type, ( floor( re_timestamp / 3600 ) * 3600 ) AS time, COUNT(re_event_type) AS num
						FROM rfid_event
						INNER JOIN rfid_citymesh ON rc_code = re_code AND rc_com_id = re_com_id 
						WHERE rc_type = {$rc_type} AND re_timestamp >= {$start_time} AND re_timestamp <= {$end_time} AND {$filter}
						GROUP BY ( floor( re_timestamp / 3600 ) * 3600 ), re_code, '{$type} ' || to_hex(re_code)::text ) AS {$fixed_type} ";
		
		$full_sql = "SELECT '{$type} ' || to_hex(base_serie.rc_code)::text AS type, base_serie.time, COALESCE({$fixed_type}.num,0) AS total, {$priority} AS priority
					FROM {$series_sql}
					LEFT JOIN 
					{$real_sql}
					ON base_serie.time = {$fixed_type}.time AND base_serie.rc_code = {$fixed_type}.rc_code";
						
		return $full_sql;
	}
	
	public function get_graph_data( $filter_company, $start_time, $end_time )
	{
		$com_filter	= self::get_company_filter( $filter_company );
		$com_filter	= '' != $com_filter ? ' AND ' . $com_filter : '';
		
		$total_entry_sql	= $this->gen_graph_subquery( 'Ingresos Totales', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 1 OR re_event_type = 5 ) {$com_filter}",
									1000 );
		
		$total_leave_sql	= $this->gen_graph_subquery( 'Egresos Totales', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 2 OR re_event_type = 6 ) {$com_filter}",
									999 );
		
		$ac_entry_sql	= $this->gen_graph_subquery( 'Ingresos Aceptados', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 1 OR re_event_type = 5 ) AND re_state = 'ACCEPTED' {$com_filter}",
									998 );
		
		$ac_leave_sql	= $this->gen_graph_subquery( 'Egresos Aceptados', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 2 OR re_event_type = 6 ) AND re_state = 'ACCEPTED' {$com_filter}",
									997 );
		
		$rc_entry_sql	= $this->gen_graph_subquery( 'Ingresos Rechazados', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 1 OR re_event_type = 5 ) AND re_state = 'REJECTED' {$com_filter}",
									996 );
		
		$rc_leave_sql	= $this->gen_graph_subquery( 'Egresos Rechazados', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 2 OR re_event_type = 6 ) AND re_state = 'REJECTED' {$com_filter}",
									995 );
		
		
		$total_vip_entry_sql	= $this->gen_graph_subquery( 'Ingresos VIP', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 5 ) {$com_filter}",
									994 );
		
		$total_vip_leave_sql	= $this->gen_graph_subquery( 'Egresos VIP', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 6 ) {$com_filter}",
									993 );
		
		$total_code_entry_sql	= $this->gen_graph_subquery_hw_code( 'Ingresos', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 1 OR re_event_type = 5 ) {$com_filter}",
									0,
									992 );
		
		$total_code_leave_sql	= $this->gen_graph_subquery_hw_code( 'Egresos', 
									$start_time, 
									$end_time, 
									"re_packet_type = 1 AND ( re_event_type = 2 OR re_event_type = 6 ) {$com_filter}",
									1, 
									991 );
		
		$sql = "( {$total_entry_sql} ) UNION ALL 
				( {$total_leave_sql} ) UNION ALL 
				( {$ac_entry_sql} ) UNION ALL 
				( {$ac_leave_sql} ) UNION ALL 
				( {$rc_entry_sql} ) UNION ALL 
				( {$rc_leave_sql} ) UNION ALL 
				( {$total_vip_entry_sql} ) UNION ALL 
				( {$total_vip_leave_sql} ) UNION ALL 
				( {$total_code_entry_sql} ) UNION ALL 
				( {$total_code_leave_sql} ) 
				ORDER BY time ASC, priority DESC, type ASC";
		
		return $this->db->get_results( $sql, ARRAY_A );
	}
	
	public function get_graph_full_data( $filter_company, $start_time, $end_time )
	{
		$start_time		= ( floor( 	$start_time / 3600 ) * 3600 );
		$end_time		= ( floor( $end_time / 3600 ) * 3600 );
		$data			= $this->get_graph_data( $filter_company, $start_time, $end_time );
		$time_count		= array_count_unique_subkey( $data, 'time' );
		
		return array( 
			'data' => $data,
			'time_count' => $time_count,
			'type_count' => array_count_unique_subkey( $data, 'type' ),
			'date_str' => strftime ('%d/%m/%Y %H:%M',$end_time ),
			'hours' => $time_count - 1
		);
	}
}
