package com.ensoft.aeir.ui.adapter;

import android.content.Context;
import android.support.annotation.NonNull;
import android.support.v7.widget.RecyclerView;
import android.text.Html;
import android.view.LayoutInflater;
import android.view.View;
import android.view.ViewGroup;
import android.widget.Button;
import android.widget.TextView;

import com.afollestad.materialdialogs.DialogAction;
import com.afollestad.materialdialogs.MaterialDialog;
import com.ensoft.aeir.App;
import com.ensoft.aeir.R;
import com.ensoft.aeir.domain.model.PatientModel;
import com.ensoft.aeir.infrastructure.helper.IntentHelper;
import com.ensoft.aeir.infrastructure.listener.AuditListener;
import com.ensoft.aeir.infrastructure.service.DistanceService;
import com.ensoft.aeir.infrastructure.service.LocationService;

import java.util.Arrays;

import butterknife.Bind;
import butterknife.ButterKnife;

public class PatientsAdapter extends RecyclerView.Adapter<PatientsAdapter.ItemViewHolder>
{
	protected PatientModel[] items;
	protected AuditListener auditListener;
	protected Context context;
	
	public PatientsAdapter( PatientModel[] items, AuditListener auditListener, Context context )
	{
		this.items = items;
		this.context = context;
		this.auditListener = auditListener;
	}
	
	@Override
	public void onAttachedToRecyclerView( RecyclerView recyclerView )
	{
		super.onAttachedToRecyclerView( recyclerView );
	}
	
	@Override
	public ItemViewHolder onCreateViewHolder( ViewGroup viewGroup, int i )
	{
		View v = LayoutInflater.from( viewGroup.getContext() ).inflate( R.layout.item_patient, viewGroup, false );
		return new ItemViewHolder( v, auditListener, context );
	}
	
	@Override
	public void onBindViewHolder( ItemViewHolder personViewHolder, int i )
	{
		personViewHolder.setItem( items[ i ] );
	}
	
	public void updatePatientModel( PatientModel patientModel )
	{
		if ( null != items && items.length > 0 )
		{
			for ( int i = 0; i < items.length; i++ )
			{
				if ( items[i].getId() == patientModel.getId() )
				{
					items[i] = patientModel;
					
					notifyDataSetChanged();
				}
			}
		}
	}
	
	@Override
	public int getItemCount()
	{
		return items.length;
	}

	static class ItemViewHolder extends RecyclerView.ViewHolder
	{
		@Bind( R.id.card_title )
		protected TextView patientNameView;
		
		@Bind( R.id.card_address )
		protected TextView patientAddess;
		
		@Bind( R.id.card_phone )
		protected TextView patientPhoneView;
		
		@Bind( R.id.patient_visit_button )
		protected Button patientVisitButton;
		
		@Bind( R.id.patient_audited_button )
		protected Button patientAudited;
		
		private Context context;
		private AuditListener auditListener;

		ItemViewHolder( View itemView, AuditListener auditListener, Context context )
		{
			super( itemView );
			this.context = context;
			this.auditListener = auditListener;
			
			ButterKnife.bind( this, itemView );
		}
		
		public void setItem( final PatientModel patient )
		{
			double distance = patient.getDistanceToUser();
			
			patientAudited.setVisibility( View.GONE );
			patientVisitButton.setVisibility( View.VISIBLE );
			patientPhoneView.setVisibility( View.VISIBLE );
			patientNameView.setText( patient.getName() );
			
			String domain = "Domicilio: " + patient.getAddress();
			String distanceStr = "";
			
			if ( distance > DistanceService.DISTANCE_TRIGGER && 999999999 != distance )
			{
				patientVisitButton.setVisibility( View.INVISIBLE );
				
				distanceStr = " ( " + new DistanceService().metersToString( distance ) + " )";
			}
			else if ( 0 == App.getInstance().getPreferencesService().getLatitude() && 0 == App.getInstance().getPreferencesService().getLongitude() )
			{
				patientVisitButton.setVisibility( View.INVISIBLE );
				
				distanceStr = " ( sin información de gps )";
			}
			else if ( patient.getLatitude() == 0 && patient.getLongitude() == 0 )
			{
				patientVisitButton.setVisibility( View.INVISIBLE );
			}
			else if ( patient.getState() == 2 )
			{
				patientVisitButton.setVisibility( View.INVISIBLE );
			}
			else if ( !new LocationService( context ).isGpsEnable() )
			{
				patientVisitButton.setVisibility( View.INVISIBLE );
			}
			
			if ( patient.isAudited() )
			{
				patientAudited.setVisibility( View.VISIBLE );
				patientVisitButton.setVisibility( View.GONE );
			}
			
			String address = domain + distanceStr;
			patientAddess.setText( address );
		
			String phone = "Teléfono: " + patient.getPhone();
			patientPhoneView.setText( phone );
			
			if ( null != patient.getPhone() && !patient.getPhone().isEmpty() )
			{
				if ( patient.getPhone().contains( "," ) )
				{
					String p = "Teléfonos: " + "<br/>" + patient.getPhone().replace( ",", "<br/>" );
					
					patientPhoneView.setText( Html.fromHtml( p ) );
				}
				
				patientPhoneView.setOnClickListener( new View.OnClickListener()
				{
					@Override
					public void onClick( View v )
					{
						new MaterialDialog.Builder( context )
							.title( R.string.app_name )
							.content( "¿Desea llamar al paciente?" )
							.positiveText( "Llamar" )
							.negativeText( "Cancelar" )
							.onPositive( new MaterialDialog.SingleButtonCallback()
							{
								@Override
								public void onClick( @NonNull MaterialDialog dialog, @NonNull DialogAction which )
								{
									if ( !patient.getPhone().contains( "," ) )
									{
										IntentHelper.dialPhoneNumber( context, patient.getPhone() );
									}
									else
									{
										String[] phones = patient.getPhone().split( "," );
										
										new MaterialDialog.Builder( context )
											.title("Seleccionar número telefónico")
											.items( Arrays.asList(phones) )
											.alwaysCallSingleChoiceCallback()
											.itemsCallbackSingleChoice(0, new MaterialDialog.ListCallbackSingleChoice()
											{
												@Override
												public boolean onSelection(MaterialDialog dialog, View view, int which, CharSequence text)
												{
													return true;
												}
											})
											.onPositive( new MaterialDialog.SingleButtonCallback()
											{
												@Override
												public void onClick( @NonNull MaterialDialog dialog, @NonNull DialogAction which )
												{
													IntentHelper.dialPhoneNumber( context, patient.getPhone() );
												}
											} )
											.positiveText("LLAMAR")
											.show();
									}
								}
							} )
							.build().show();
					}
				} );
				
			}
			else
			{
				patientPhoneView.setVisibility( View.GONE );
			}
			
			patientVisitButton.setOnClickListener( new View.OnClickListener()
			{
				@Override
				public void onClick( View view )
				{
					auditListener.onStartAudit( patient );
				}
			} );
		}
	}
}