package com.ensoft.aeir.ui.activity;

import android.app.LoaderManager;
import android.content.Intent;
import android.content.Loader;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.Rect;
import android.os.Bundle;
import android.os.Handler;
import android.support.v4.app.ActivityCompat;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.util.Log;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.widget.EditText;

import com.ensoft.aeir.App;
import com.ensoft.aeir.R;
import com.ensoft.aeir.domain.model.LocationModel;
import com.ensoft.aeir.domain.model.PatientModel;
import com.ensoft.aeir.domain.table.LocationTable;
import com.ensoft.aeir.domain.table.PatientTable;
import com.ensoft.aeir.infrastructure.event.GPSEvent;
import com.ensoft.aeir.infrastructure.listener.AuditListener;
import com.ensoft.aeir.infrastructure.request.CheckTokenRequest;
import com.ensoft.aeir.infrastructure.request.PatientsGetRequest;
import com.ensoft.aeir.ui.adapter.PatientsAdapter;
import com.ensoft.restafari.network.rest.response.HttpStatus;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;

import java.util.Arrays;
import java.util.Comparator;

import butterknife.Bind;
import butterknife.ButterKnife;
import butterknife.OnFocusChange;
import butterknife.OnTextChanged;

public class PatientsActivity extends LocationAwareActivity implements
	AuditListener,
	LoaderManager.LoaderCallbacks<Cursor>
{
	public static final String TAG = PatientsActivity.class.getCanonicalName();
	public static final String SEARCH_STRING = "searchString";
	
	public static final int PATIENTS_LOADER = 1;
	public static final int LOCATION_LOADER = 2;
	
	@Bind( R.id.lista_pacientes )
	protected RecyclerView patientList;

	protected PatientsAdapter patientsAdapter;

	protected Cursor patientsCursor;
	protected Cursor locationCursor;
	
	protected long patientsGetReqId;
	protected boolean gpsInitialized = false;
	
	
	protected Handler searchTextHandler;
	protected String searchText;
	
	@Override
	protected void onCreate( Bundle savedInstanceState )
	{
		super.onCreate( savedInstanceState );

		setContentView( R.layout.activity_main );

		ButterKnife.bind( this );
	}
	
	@Override
	protected void onPostCreate( Bundle savedInstanceState )
	{
		super.onPostCreate( savedInstanceState );
		
		requestPermissions();
		
		CheckTokenRequest.addRequest();
	}

	@Override
	protected void onDestroy()
	{
		super.onDestroy();
		
		if ( null != patientsCursor && !patientsCursor.isClosed() )
			patientsCursor.close();
		
		if ( null != locationCursor && !locationCursor.isClosed() )
			locationCursor.close();
	}
	
	void initUI()
	{
		patientList.setAdapter( new PatientsAdapter( new PatientModel[] {}, this, this ) );
		patientList.setLayoutManager( new LinearLayoutManager( getApplicationContext() ) );

		getPatients();
	}
	
	@Override
	protected void onResume()
	{
		EventBus.getDefault().register( this );

		super.onResume();
		
		initUI();
	}

	@Override
	public void onPause()
	{
		EventBus.getDefault().unregister( this );

		super.onPause();
	}
	
	protected void refreshGPS()
	{
		if ( gpsInitialized )
		{
			App.getInstance().getGPSServiceProvider().updateLocation();
		}
	}
	
	protected void getPatients()
	{
		refreshGPS();
		
		getRequestReceiverService().addRequest( patientsGetReqId = PatientsGetRequest.addRequest() );

		createdDialog( "Actualizando" );
	}
	
	protected void requestPermissions()
	{
		if ( ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_FINE_LOCATION ) != PackageManager.PERMISSION_GRANTED
			|| ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_COARSE_LOCATION ) != PackageManager.PERMISSION_GRANTED )
		{
			ActivityCompat.requestPermissions( this, new String[]{ android.Manifest.permission.ACCESS_FINE_LOCATION, android.Manifest.permission.ACCESS_COARSE_LOCATION }, 100 );
		}
		else
		{
			initServices();
		}
	}

	@Override
	public void onRequestPermissionsResult( int requestCode, String permissions[], int[] grantResults )
	{
		switch ( requestCode )
		{
			case 100:
			{
				if ( grantResults.length == 0 || grantResults[ 0 ] != PackageManager.PERMISSION_GRANTED )
				{}
				else
				{
					if ( ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_FINE_LOCATION ) != PackageManager.PERMISSION_GRANTED ||
						ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_COARSE_LOCATION ) != PackageManager.PERMISSION_GRANTED )
					{
						return;
					}
					
					initServices();
				}
			}
		}
	}
	
	public void initServices()
	{
		gpsInitialized = true;
		
		App.getInstance().getGPSServiceProvider();
		
		getLoaderManager().initLoader( LOCATION_LOADER, null, this );
	}
	
	public void onLocationChanged( LocationModel location )
	{
		App.getInstance().getPreferencesService().setLocation( location.getLatitude(), location.getLongitude() );

		if ( patientsAdapter != null )
		{
			patientsAdapter.notifyDataSetChanged();
		}
	}
	
	@Override
	public boolean onCreateOptionsMenu( Menu menu )
	{
		MenuInflater inflater = getMenuInflater();
		inflater.inflate( R.menu.menu, menu );
		return true;
	}
	
	@Override
	public boolean onOptionsItemSelected( MenuItem item )
	{
		switch ( item.getItemId() )
		{
			case R.id.new_game:
				startActivity( new Intent( this, ConfigActivity.class ) );
				return true;
			case R.id.reload:
				getPatients();
			default:
				return super.onOptionsItemSelected( item );
		}
	}
	
	@Override
	public void onStartAudit( final PatientModel patient )
	{
		AuditActivity.newInstance( this, patient.getPatientId() );
	}

	@Override
	public void onRequestSuccess( long requestId )
	{
		if ( requestId == patientsGetReqId )
		{
			getLoaderManager().initLoader( PATIENTS_LOADER, null, this );
		}
		
		closeDialog();
	}
	
	@Override
	public void onRequestError( long requestId, int resultCode, String resultMsg )
	{
		if ( resultCode == HttpStatus.UNAUTHORIZED_401.getCode() )
		{
			App.getInstance().getPreferencesService().setUser( null );
			
			startActivity( new Intent( this, LoginActivity.class ).setFlags( Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK ) );
		}

		closeDialog();
		
		errorMessageFromResponse( resultMsg );
	}
	
	@Override
	public Loader<Cursor> onCreateLoader( int id, Bundle args )
	{
		if ( id == PATIENTS_LOADER )
		{
			String name = null != args ? args.getString( SEARCH_STRING ) : null;
			
			return new PatientTable().getAll( name );
		}
		else if ( id == LOCATION_LOADER )
			return new LocationTable().getLoaderFromId( App.getInstance().getPreferencesService().getUser().getId() );
		
		return null;
	}
	
	protected void refreshPatients()
	{
		if ( null != patientsCursor && patientsCursor.moveToFirst() )
		{
			final PatientModel[] patients = new PatientModel[ patientsCursor.getCount() ];
			
			int i = 0;
			
			do
			{
				patients[ i ] = new PatientModel();
				patients[ i ].fromCursor( patientsCursor );
				i++;
			}
			while ( patientsCursor.moveToNext() );
			
			if ( 0 != App.getInstance().getPreferencesService().getLatitude() || 0 != App.getInstance().getPreferencesService().getLongitude() )
			{
				Arrays.sort( patients, new Comparator<PatientModel>()
				{
					@Override
					public int compare( PatientModel o1, PatientModel o2 )
					{
						if ( o1.getAudited() < o2.getAudited() )
						{
							return -1;
						}
						else if ( o1.getAudited() > o2.getAudited() )
						{
							return 1;
						}
						
						return o1.getDistanceToUser() < o2.getDistanceToUser() ? -1 : 1;
					}
				} );
			}
			
			runOnUiThread( new Runnable()
			{
				@Override
				public void run()
				{
					patientsAdapter = new PatientsAdapter( patients, PatientsActivity.this, PatientsActivity.this );
					
					patientList.setAdapter( patientsAdapter );
				}
			} );
		}
	}
	
	@Override
	public void onLoadFinished( Loader<Cursor> loader, Cursor data )
	{
		if ( loader.getId() == PATIENTS_LOADER )
		{
			patientsCursor = data;
			
			refreshPatients();
		}
		else if ( loader.getId() == LOCATION_LOADER )
		{
			locationCursor = data;
			
			if ( null != locationCursor && locationCursor.moveToFirst() )
			{
				LocationModel locationModel = (LocationModel)new LocationModel().fromCursor( locationCursor );
				
				onLocationChanged( locationModel );
			}
		}
	}
	
	@Override
	public void onLoaderReset( Loader<Cursor> loader )
	{
	}

	@Subscribe
	public void onEvent( GPSEvent.onLocationChangeEvent event )
	{
		final String log = "onEvent: " + event.latitude + " " + event.longitude;

		Log.d( TAG, log );
	}
	
	protected Runnable onSearchText = new Runnable()
	{
		@Override
		public void run()
		{
			Bundle data = new Bundle();
			data.putString( SEARCH_STRING, searchText );
			getLoaderManager().restartLoader( PATIENTS_LOADER, data, PatientsActivity.this );
		}
	};
	
	@OnFocusChange( R.id.item_checker_search_edit_text )
	protected void onSearchChanged( View view, boolean hasFocus )
	{
		if ( !hasFocus )
		{
			hideSoftKeyboard();
			
			patientList.requestFocus();
		}
	}
	
	@OnTextChanged( R.id.item_checker_search_edit_text )
	protected void onSearchTextChanged( final CharSequence text )
	{
		searchText = text.toString();
		
		if ( null == searchTextHandler )
		{
			searchTextHandler = new Handler();
		}
		else
		{
			searchTextHandler.removeCallbacks( onSearchText );
		}
		
		searchTextHandler.postDelayed( onSearchText, 500 );
	}
	
	@Override
	public boolean dispatchTouchEvent(MotionEvent ev)
	{
		if ( ev.getAction() == MotionEvent.ACTION_DOWN )
		{
			View view = getCurrentFocus();
			if ( view != null && view instanceof EditText )
			{
				Rect r = new Rect();
				view.getGlobalVisibleRect(r);
				int rawX = (int)ev.getRawX();
				int rawY = (int)ev.getRawY();
				
				if (!r.contains(rawX, rawY))
				{
					view.clearFocus();
					
					hideSoftKeyboard();
				}
			}
		}
		
		return super.dispatchTouchEvent(ev);
	}
}
