package com.ensoft.aeir.infrastructure.service;

import android.content.Context;
import android.content.Intent;
import android.location.Location;
import android.location.LocationListener;
import android.location.LocationManager;
import android.os.Bundle;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;

import com.app.ensoft.medicontrol.infrastructure.service.IGPSService;
import com.app.ensoft.medicontrol.infrastructure.service.IGPSServiceCallback;
import com.ensoft.aeir.App;
import com.ensoft.aeir.domain.model.UserModel;
import com.ensoft.aeir.domain.table.LocationTable;

public class GPSService extends ServiceWithCallbacks<IGPSServiceCallback>
{
	public static final String TAG = GPSService.class.getCanonicalName();

	private static final long sMinTimeMillis = 2000;
	private static final long sMinDistanceMeters = 100;
	public static final float GPS_MIN_ACCURACY_METERS = 100;

	private LocationManager mLocationManager;
	private LocationListener mLocationListener;

	public static Location getLastKnownLocation( Context context )
	{
		try
		{
			LocationManager locationManager = (LocationManager)context.getSystemService(Context.LOCATION_SERVICE);

			return locationManager.getLastKnownLocation( LocationManager.PASSIVE_PROVIDER );
		}
		catch ( SecurityException securityException )
		{
			return null;
		}
		catch ( Exception e )
		{
			return null;
		}
	}

	private final IGPSService.Stub mBinder = new IGPSService.Stub()
	{
		@Override
		public void updateLocation()
		{
			try
			{
				Log.d( TAG, "IGPSService.updateLocation called" );

				Location location = getLastKnownLocation( GPSService.this );

				if ( null != location )
					mLocationListener.onLocationChanged( location );
			}
			catch ( SecurityException securityException )
			{
			}
		}

		@Override
		public void registerCallback(IGPSServiceCallback callback) throws RemoteException
		{
			addCallback(callback);
		}

		@Override
		public void unregisterCallback(IGPSServiceCallback callback) throws RemoteException
		{
			removeCallback(callback);
		}
	};

	private void startLoggerService()
	{
		Log.w( TAG, "Starting GPSService" );
		
		mLocationManager = (LocationManager) getSystemService( Context.LOCATION_SERVICE );

		mLocationListener = createLocationListener();

		try
		{
			Location location = getLastKnownLocation( this );
			
			if ( null != location )
				mLocationListener.onLocationChanged( location );

			mLocationManager.requestLocationUpdates( LocationManager.GPS_PROVIDER, sMinTimeMillis, sMinDistanceMeters, mLocationListener );
		}
		catch ( SecurityException securityException )
		{
		}

		Log.w( TAG, "GPSService Started" );
	}

	protected GPSLocationListener createLocationListener()
	{
		return new GPSLocationListener();
	}

	private void shutdownLoggerService()
	{
		Log.w( TAG, "GPSService Ended" );
		
		mLocationManager.removeUpdates( mLocationListener );
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId)
	{
		super.onStartCommand( intent, flags, startId );
		return START_STICKY;
	}

	@Override
	public void onCreate()
	{
		super.onCreate();

		startLoggerService();
	}
	
	@Override
	public void onDestroy()
	{
		super.onDestroy();

		shutdownLoggerService();
	}

	protected void sendPositionToClient( final Location loc )
	{
		broadcastCallback( new CallbackBroadcaster()
		{
			@Override
			public void run( IGPSServiceCallback item )
			{
				try
				{
					item.onLocationChanged( loc.getLatitude(), loc.getLongitude() );
				}
				catch ( RemoteException e )
				{}
			}
		} );
	}

	@Override
	public IBinder onBind( Intent intent )
	{
		return mBinder;
	}

	public class GPSLocationListener implements LocationListener
	{
		public void onLocationChanged( Location loc )
		{
			Log.w( TAG, "Location changed!\nAccuracy: " + String.valueOf( loc.getAccuracy() ) + " and minimum accuracy is set as " + String.valueOf( GPS_MIN_ACCURACY_METERS ) );

			if ( loc.hasAccuracy() && loc.getAccuracy() <= GPS_MIN_ACCURACY_METERS )
			{
				App.getInstance().getPreferencesService().setLocation( loc );
				
				UserModel userModel = App.getInstance().getPreferencesService().getUser();
				
				if ( null != userModel )
				{
					Log.w( TAG, "Saving location in database for user id: " + userModel.getId() );
					
					new LocationTable().save( userModel.getId(), loc );
				}
				else
				{
					Log.e( TAG, "UserModel not available." );
				}

				sendPositionToClient( loc );
			}
		}

		public void onProviderDisabled( String provider )
		{
		}

		public void onProviderEnabled( String provider )
		{
		}

		public void onStatusChanged( String provider, int status, Bundle extras )
		{
		}
	}
}
