package com.ensoft.medicontrol.ui.activity;

import android.app.Activity;
import android.app.LoaderManager;
import android.content.Intent;
import android.content.Loader;
import android.content.pm.PackageManager;
import android.database.Cursor;
import android.graphics.Rect;
import android.os.Bundle;
import android.os.Handler;
import android.support.annotation.NonNull;
import android.support.v4.app.ActivityCompat;
import android.support.v4.app.DialogFragment;
import android.support.v7.widget.LinearLayoutManager;
import android.support.v7.widget.RecyclerView;
import android.text.InputType;
import android.util.Log;
import android.view.Gravity;
import android.view.Menu;
import android.view.MenuInflater;
import android.view.MenuItem;
import android.view.MotionEvent;
import android.view.View;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.afollestad.materialdialogs.DialogAction;
import com.afollestad.materialdialogs.MaterialDialog;
import com.ensoft.medicontrol.App;
import com.ensoft.medicontrol.R;
import com.ensoft.medicontrol.domain.model.LocationModel;
import com.ensoft.medicontrol.domain.model.PatientModel;
import com.ensoft.medicontrol.domain.model.VisitModel;
import com.ensoft.medicontrol.domain.table.LocationTable;
import com.ensoft.medicontrol.domain.table.PatientTable;
import com.ensoft.medicontrol.domain.table.VisitTable;
import com.ensoft.medicontrol.infrastructure.event.GPSEvent;
import com.ensoft.medicontrol.infrastructure.listener.PatientListener;
import com.ensoft.medicontrol.infrastructure.provider.DelayedTask;
import com.ensoft.medicontrol.infrastructure.request.CheckTokenRequest;
import com.ensoft.medicontrol.infrastructure.request.PatientsGetRequest;
import com.ensoft.medicontrol.infrastructure.request.VisitEndRequest;
import com.ensoft.medicontrol.infrastructure.request.VisitStartRequest;
import com.ensoft.medicontrol.infrastructure.service.DateService;
import com.ensoft.medicontrol.infrastructure.service.DelayedTaskService;
import com.ensoft.medicontrol.infrastructure.service.LocationService;
import com.ensoft.medicontrol.ui.adapter.PatientsAdapter;
import com.ensoft.medicontrol.ui.dialog.VisitEndDialog;
import com.ensoft.restafari.network.rest.response.HttpStatus;

import org.greenrobot.eventbus.EventBus;
import org.greenrobot.eventbus.Subscribe;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Timer;
import java.util.TimerTask;

import butterknife.Bind;
import butterknife.ButterKnife;
import butterknife.OnFocusChange;
import butterknife.OnTextChanged;

public class PatientsActivity extends LocationAwareActivity implements
	PatientListener,
	LoaderManager.LoaderCallbacks<Cursor>
{
	public static final String TAG = PatientsActivity.class.getCanonicalName();
	public static final String SEARCH_STRING = "searchString";
	
	public static final int PATIENTS_LOADER = 1;
	public static final int ACTIVE_VISIT_LOADER = 2;
	public static final int LOCATION_LOADER = 3;
	
	@Bind( R.id.lista_pacientes )
	protected RecyclerView patientList;

	@Bind( R.id.timeView )
	protected TextView time;

	@Bind( R.id.time_container )
	protected LinearLayout timeContainer;

	protected PatientsAdapter patientsAdapter;

	protected Timer timer;
	protected TimerTask timerTask;

	protected Cursor patientsCursor;
	protected Cursor activeVisitCursor;
	protected Cursor locationCursor;
	
	protected long patientsGetReqId;
	protected long visitStartReqId;
	protected long visitEndReqId;
	protected VisitModel currentVisit;
	protected boolean gpsInitialized = false;
	protected DelayedTaskService delayedStart = new DelayedTaskService();
	
	protected Handler searchTextHandler;
	protected String searchText;
	
	@Override
	protected void onCreate( Bundle savedInstanceState )
	{
		super.onCreate( savedInstanceState );

		setContentView( R.layout.activity_main );

		ButterKnife.bind( this );
	}
	
	@Override
	protected void onPostCreate( Bundle savedInstanceState )
	{
		super.onPostCreate( savedInstanceState );
		
		requestPermissions();
		
		getLoaderManager().initLoader( ACTIVE_VISIT_LOADER, null, this );
		
		CheckTokenRequest.addRequest();
	}

	@Override
	protected void onDestroy()
	{
		super.onDestroy();
		
		if ( null != patientsCursor && !patientsCursor.isClosed() )
			patientsCursor.close();
		
		if ( null != activeVisitCursor && !activeVisitCursor.isClosed() )
			activeVisitCursor.close();
		
		if ( null != locationCursor && !locationCursor.isClosed() )
			locationCursor.close();
	}
	
	void initUI()
	{
		patientList.setAdapter( new PatientsAdapter( new PatientModel[] {}, this, this ) );
		patientList.setLayoutManager( new LinearLayoutManager( getApplicationContext() ) );

		timeContainer.setVisibility( View.GONE );

		getPatients();
	}
	
	@Override
	protected void onResume()
	{
		EventBus.getDefault().register( this );

		super.onResume();
		
		initUI();
	}

	@Override
	public void onPause()
	{
		EventBus.getDefault().unregister( this );

		super.onPause();
	}
	
	protected void refreshGPS()
	{
		if ( gpsInitialized )
		{
			App.getInstance().getGPSServiceProvider().updateLocation();
		}
	}
	
	protected void getPatients()
	{
		refreshGPS();
		
		getRequestReceiverService().addRequest( patientsGetReqId = PatientsGetRequest.addRequest() );

		createdDialog( "Actualizando" );
	}
	
	protected void requestPermissions()
	{
		if ( ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_FINE_LOCATION ) != PackageManager.PERMISSION_GRANTED
			|| ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_COARSE_LOCATION ) != PackageManager.PERMISSION_GRANTED )
		{
			ActivityCompat.requestPermissions( this, new String[]{ android.Manifest.permission.ACCESS_FINE_LOCATION, android.Manifest.permission.ACCESS_COARSE_LOCATION }, 100 );
		}
		else
		{
			initServices();
		}
	}

	@Override
	public void onRequestPermissionsResult( int requestCode, String permissions[], int[] grantResults )
	{
		switch ( requestCode )
		{
			case 100:
			{
				
				if ( grantResults.length == 0 || grantResults[ 0 ] != PackageManager.PERMISSION_GRANTED )
				{
					
				}
				else
				{
					if ( ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_FINE_LOCATION ) != PackageManager.PERMISSION_GRANTED ||
						ActivityCompat.checkSelfPermission( this, android.Manifest.permission.ACCESS_COARSE_LOCATION ) != PackageManager.PERMISSION_GRANTED )
					{
						return;
					}
					
					initServices();
				}
			}
		}
	}
	
	public void initServices()
	{
		gpsInitialized = true;
		
		App.getInstance().getGPSServiceProvider();
		
		getLoaderManager().initLoader( LOCATION_LOADER, null, this );
	}
	
	public void onLocationChanged( LocationModel location )
	{
		App.getInstance().getPreferencesService().setLocation( location.getLatitude(), location.getLongitude() );

		if ( patientsAdapter != null )
		{
			patientsAdapter.notifyDataSetChanged();
		}
	}
	
	@Override
	public boolean onCreateOptionsMenu( Menu menu )
	{
		MenuInflater inflater = getMenuInflater();
		inflater.inflate( R.menu.menu, menu );
		return true;
	}
	
	@Override
	public boolean onOptionsItemSelected( MenuItem item )
	{
		switch ( item.getItemId() )
		{
			case R.id.new_game:
				startActivity( new Intent( this, ConfigActivity.class ) );
				return true;
			case R.id.reload:
				getPatients();
			default:
				return super.onOptionsItemSelected( item );
		}
	}
	
	void saveVisit( final PatientModel item )
	{
		createdDialog( "Iniciando visita" );
		
		getRequestReceiverService().addRequest( visitStartReqId = VisitStartRequest.addRequest( item.getPatientId() ) );
	}
	
	void endVisit( String observations )
	{
		createdDialog( "Finalizando visita" );
		
		getRequestReceiverService().addRequest( visitEndReqId = VisitEndRequest.addRequest( observations ) );
	}
	
	protected void updateTimeContainerVisibility()
	{
		runOnUiThread( new Runnable()
		{
			@Override
			public void run()
			{
				timeContainer.setVisibility( currentVisit != null ? View.VISIBLE : View.GONE );
			}
		} );
	}
	
	protected void cancelTimer()
	{
		if ( null != timer )
		{
			timer.cancel();
			timer = null;
		}
		
		if ( null != timerTask )
		{
			timerTask.cancel();
		}
		
		updateTimeContainerVisibility();
		
		refreshPatients();
	}
	
	public void startTimer( final PatientModel patient )
	{
		if ( null != timer )
		{
			timer.cancel();
			timer = null;
		}
		
		if ( null != timerTask )
		{
			timerTask.cancel();
		}
		
		timer = new Timer();
		
		timerTask = new TimerTask()
		{
			public void run()
			{
				final DateService dateService = new DateService();
				final VisitModel visit = patient.getCurrentVisit();
				final long diff = Math.max( 0, dateService.getCurrentTimestamp() - visit.getStartTimestampFixed() );
				
				if ( patient.getSessionType() == 0 && visit.expired() )
				{
					if ( null != timer )
					{
						timer.cancel();
						timer = null;
					}
					
					runOnUiThread( new Runnable()
					{
						@Override
						public void run()
						{
							if ( null != patientsAdapter )
							{
								currentVisit = null;
								
								patient.setCurrentVisit( null );
								
								patientsAdapter.updatePatientModel( patient );
							}
						}
					} );
				}
				else
				{
					runOnUiThread( new Runnable()
					{
						@Override
						public void run()
						{
							time.setText( "Tiempo de llegada: " + dateService.getTimeString( visit.getStartTimestampFixed() ) + "\nTiempo transcurrido: " + dateService.secondsToString( diff ) );
						}
					} );
				}
				
				updateTimeContainerVisibility();
			}
		};
		
		timer.schedule( timerTask, 100, 1000 );
	}
	
	@Override
	public void onSaveVisit( final PatientModel patient )
	{
		LocationService locationService = new LocationService( this );

		if ( locationService.isEnable() )
		{
			saveVisit( patient );
		}
		else
		{
			delayedStart.add( new DelayedTask()
			{
				@Override
				public void run()
				{
					saveVisit( patient );
				}
			} );

			locationService.requestEnableLocation( this );
		}
	}

	@Override
	protected void onActivityResult( int requestCode, int resultCode, Intent data )
	{
		if ( requestCode == LocationService.REQUEST_LOCATION )
		{
			if ( Activity.RESULT_OK == resultCode )
			{
				delayedStart.run();
			}
		}
	}

	@Override
	public void onEndVisit( final PatientModel patient )
	{
		VisitEndDialog.newDialog( this, new MaterialDialog.SingleButtonCallback()
		{
			@Override
			public void onClick( @NonNull MaterialDialog dialog, @NonNull DialogAction which )
			{
				String observations = dialog.getInputEditText().getText().toString();

				endVisit( observations );
			}
		} );
	}
	
	@Override
	public void onRequestSuccess( long requestId )
	{
		if ( requestId == patientsGetReqId )
		{
			getLoaderManager().initLoader( PATIENTS_LOADER, null, this );
		}
		else if ( requestId == visitStartReqId )
		{
			
		}
		else if ( requestId == visitEndReqId )
		{
			
		}
		
		closeDialog();
	}
	
	@Override
	public void onRequestError( long requestId, int resultCode, String resultMsg )
	{
		if ( resultCode == HttpStatus.UNAUTHORIZED_401.getCode() )
		{
			App.getInstance().getPreferencesService().setUser( null );
			
			startActivity( new Intent( this, LoginActivity.class ).setFlags( Intent.FLAG_ACTIVITY_NEW_TASK | Intent.FLAG_ACTIVITY_CLEAR_TASK ) );
		}

		closeDialog();
		
		errorMessageFromResponse( resultMsg );
	}
	
	@Override
	public Loader<Cursor> onCreateLoader( int id, Bundle args )
	{
		if ( id == PATIENTS_LOADER )
		{
			String name = null != args ? args.getString( SEARCH_STRING ) : null;
			
			return new PatientTable().getAll( name );
		}
		else if ( id == ACTIVE_VISIT_LOADER )
			return new VisitTable().getLoaderFromProfessional( App.getInstance().getPreferencesService().getUser().getId() );
		else if ( id == LOCATION_LOADER )
			return new LocationTable().getLoaderFromId( App.getInstance().getPreferencesService().getUser().getId() );
		
		return null;
	}
	
	protected void refreshPatients()
	{
		if ( null != patientsCursor && patientsCursor.moveToFirst() )
		{
			final PatientModel[] patients = new PatientModel[ patientsCursor.getCount() ];
			
			int i = 0;
			
			do
			{
				patients[ i ] = new PatientModel();
				patients[ i ].fromCursor( patientsCursor );
				
				final PatientModel patient = patients[ i ];
				
				if ( null != currentVisit && !currentVisit.expired() && currentVisit.getPatientId() == patient.getPatientId() )
				{
					patient.setCurrentVisit( currentVisit );
					
					startTimer( patient );
					
					App.getInstance().getNotificationServiceProvider().createVisit(
						patient.getName(),
						currentVisit.getStartTimestampFixed() * 1000,
						currentVisit.getLength()
					);
					
					if ( null != patientsAdapter )
					{
						runOnUiThread( new Runnable()
						{
							@Override
							public void run()
							{
								patientsAdapter.updatePatientModel( patient );
							}
						} );
					}
				}
				
				i++;
			}
			while ( patientsCursor.moveToNext() );
			
			if ( 0 != App.getInstance().getPreferencesService().getLatitude() || 0 != App.getInstance().getPreferencesService().getLongitude() )
			{
				Arrays.sort( patients, new Comparator<PatientModel>()
				{
					@Override
					public int compare( PatientModel o1, PatientModel o2 )
					{
						return o1.getDistanceToUser() < o2.getDistanceToUser() ? -1 : 1;
					}
				} );
			}
			
			if ( null == currentVisit )
				updateTimeContainerVisibility();
			
			runOnUiThread( new Runnable()
			{
				@Override
				public void run()
				{
					patientsAdapter = new PatientsAdapter( patients, PatientsActivity.this, PatientsActivity.this );
					
					patientList.setAdapter( patientsAdapter );
				}
			} );
		}
	}
	
	@Override
	public void onLoadFinished( Loader<Cursor> loader, Cursor data )
	{
		if ( loader.getId() == PATIENTS_LOADER )
		{
			patientsCursor = data;
			
			refreshPatients();
		}
		else if ( loader.getId() == ACTIVE_VISIT_LOADER )
		{
			activeVisitCursor = data;
			
			if ( null != activeVisitCursor && activeVisitCursor.moveToFirst() )
			{
				currentVisit = (VisitModel)new VisitModel().fromCursor( activeVisitCursor );
				
				refreshPatients();
			}
			else
			{
				currentVisit = null;
				
				cancelTimer();
			}
			
			new Handler( ).postDelayed( new Runnable()
			{
				@Override
				public void run()
				{
					runOnUiThread( new Runnable()
					{
						@Override
						public void run()
						{
							if ( null != currentVisit && timeContainer.getVisibility() == View.GONE )
							{
								refreshPatients();
							}
						}
					} );
				}
			}, 1000 );
		}
		else if ( loader.getId() == LOCATION_LOADER )
		{
			locationCursor = data;
			
			if ( null != locationCursor && locationCursor.moveToFirst() )
			{
				LocationModel locationModel = (LocationModel)new LocationModel().fromCursor( locationCursor );
				
				onLocationChanged( locationModel );
			}
		}
	}
	
	@Override
	public void onLoaderReset( Loader<Cursor> loader )
	{
	}

	@Subscribe
	public void onEvent( GPSEvent.onLocationChangeEvent event )
	{
		final String log = "onEvent: " + event.latitude + " " + event.longitude;

		Log.d( TAG, log );
	}
	
	protected Runnable onSearchText = new Runnable()
	{
		@Override
		public void run()
		{
			Bundle data = new Bundle();
			data.putString( SEARCH_STRING, searchText );
			getLoaderManager().restartLoader( PATIENTS_LOADER, data, PatientsActivity.this );
		}
	};
	
	@OnFocusChange( R.id.item_checker_search_edit_text )
	protected void onSearchChanged( View view, boolean hasFocus )
	{
		if ( !hasFocus )
		{
			hideSoftKeyboard();
			
			patientList.requestFocus();
		}
	}
	
	@OnTextChanged( R.id.item_checker_search_edit_text )
	protected void onSearchTextChanged( final CharSequence text )
	{
		searchText = text.toString();
		
		if ( null == searchTextHandler )
		{
			searchTextHandler = new Handler();
		}
		else
		{
			searchTextHandler.removeCallbacks( onSearchText );
		}
		
		searchTextHandler.postDelayed( onSearchText, 500 );
	}
	
	@Override
	public boolean dispatchTouchEvent(MotionEvent ev)
	{
		if ( ev.getAction() == MotionEvent.ACTION_DOWN )
		{
			View view = getCurrentFocus();
			if ( view != null && view instanceof EditText )
			{
				Rect r = new Rect();
				view.getGlobalVisibleRect(r);
				int rawX = (int)ev.getRawX();
				int rawY = (int)ev.getRawY();
				
				if (!r.contains(rawX, rawY))
				{
					view.clearFocus();
					
					hideSoftKeyboard();
				}
			}
		}
		
		return super.dispatchTouchEvent(ev);
	}
}
