package com.ensoft.medicontrol.infrastructure.service;

import android.app.AlarmManager;
import android.app.Notification;
import android.app.PendingIntent;
import android.content.BroadcastReceiver;
import android.content.Context;
import android.content.Intent;
import android.content.IntentFilter;
import android.location.Location;
import android.os.Build;
import android.os.IBinder;
import android.os.RemoteException;
import android.util.Log;

import com.app.ensoft.medicontrol.infrastructure.service.INotificationService;
import com.app.ensoft.medicontrol.infrastructure.service.INotificationServiceCallback;
import com.ensoft.medicontrol.App;
import com.ensoft.medicontrol.domain.model.PatientModel;
import com.ensoft.medicontrol.domain.model.UserModel;
import com.ensoft.medicontrol.domain.model.VisitModel;
import com.ensoft.medicontrol.domain.table.PatientTable;
import com.ensoft.medicontrol.domain.table.VisitTable;
import com.ensoft.medicontrol.infrastructure.request.AlarmVisitEndRequest;
import com.ensoft.medicontrol.infrastructure.request.LocationUpdateRequest;

import java.util.Calendar;

public class NotificationService extends ServiceWithCallbacks<INotificationServiceCallback>
{
	private static final String TAG = NotificationService.class.getCanonicalName();
	private static final String ACTION_VISIT_START_NOTIFICATION 	= "com.ensoft.medicontrol.infrastructure.service.NotificationService.action.createVisit";
	private static final String ACTION_VISIT_ENDED_NOTIFICATION 	= "com.ensoft.medicontrol.infrastructure.service.NotificationService.action.visitEnded";
	private static final String ACTION_VISIT_CANCEL_NOTIFICATION 	= "com.ensoft.medicontrol.infrastructure.service.NotificationService.action.visitCancel";
	private static final String ACTION_VISIT_END 					= "com.ensoft.medicontrol.infrastructure.service.NotificationService.action.endVisit";
	private static final String ACTION_VISIT_REFRESH 				= "com.ensoft.medicontrol.infrastructure.service.NotificationService.action.refreshVisit";
	private static final int ALARM_REFRESH_SECS 					= 60 * 10;
	
	protected BroadcastReceiver broadcastReceiver = new BroadcastReceiver()
	{
		@Override
		public void onReceive( Context context, Intent intent )
		{
			Log.w( TAG, "Received broadcast." );

			if ( intent.getAction().equals( ACTION_VISIT_END ) )
			{
				Log.w( TAG, "Received broadcast to end visit." );
				
				visitEnd();
			}
			else if ( intent.getAction().equals( ACTION_VISIT_REFRESH ) )
			{
				Log.w( TAG, "Received broadcast to refresh visit." );

				try
				{
					UserModel userModel = App.getInstance().getPreferencesService().getUser();

					if ( null != userModel )
					{
						VisitTable visitTable = new VisitTable();
						VisitModel[] visits = visitTable.getVisits( userModel.getId() );

						if ( null != visits && visits.length > 0 && new LocationService( NotificationService.this ).isGpsEnable() )
						{
							Location loc = GPSService.getLastKnownLocation( NotificationService.this );

							if ( null != loc && loc.getAccuracy() <= GPSService.GPS_MIN_ACCURACY_METERS )
							{
								LocationUpdateRequest.addRequest( loc.getLatitude(), loc.getLongitude() );
							}

							if ( new DateService().getCurrentTimestamp() + ALARM_REFRESH_SECS <= visits[0].getEndTimestampFixed() )
							{
								Log.d( TAG, "Alarm refresh re-set." );

								setAlarm( ALARM_REFRESH_SECS, getAliveIntent() );
							}
						}
						else
						{
							Log.d( TAG, "Cancelled alive alarm" );

							cancelAliveAlarm();
						}
					}
					else
					{
						Log.d( TAG, "Cancelled alive alarm" );

						cancelAliveAlarm();
					}
				}
				catch ( Exception exception )
				{
					Log.w( TAG, exception.toString() );
				}
			}
		}
	};

	private final INotificationService.Stub mBinder = new INotificationService.Stub()
	{
		@Override
		public void createVisit( String name, long when, long length ) throws RemoteException
		{
			NotificationService.this.createVisit( name, when, length );
		}

		@Override
		public void visitEnded( String title, String name ) throws RemoteException
		{
			NotificationService.this.visitEnded( title, name );
		}

		@Override
		public void visitCancel() throws RemoteException
		{
			NotificationService.this.visitCancel();
		}

		@Override
		public void registerCallback(INotificationServiceCallback callback) throws RemoteException
		{
			addCallback(callback);
		}

		@Override
		public void unregisterCallback(INotificationServiceCallback callback) throws RemoteException
		{
			removeCallback(callback);
		}
	};

	private void startLoggerService()
	{
		Log.w( TAG, "NotificationService Starting" );
		
		registerReceiver( broadcastReceiver, new IntentFilter( ACTION_VISIT_END ) );
		registerReceiver( broadcastReceiver, new IntentFilter( ACTION_VISIT_REFRESH ) );

		UserModel user = App.getInstance().getPreferencesService().getUser();

		if ( null != user )
		{
			VisitTable visitTable = new VisitTable();
			VisitModel[] visits = visitTable.getVisits( user.getId() );

			if ( null != visits && visits.length > 0 )
			{
				for ( VisitModel visit : visits  )
				{
					if ( !visit.expired() )
					{
						PatientModel patientModel = new PatientTable().getModelFromId( visit.getPatientId() );

						if ( null != patientModel )
						{
							Log.w( TAG, "Recovering visit notification." );

							createVisit( patientModel.getName(),
								visit.getStartTimestampFixed() * 1000,
								visit.getLength()
							);
						}
					}
				}
			}
		}
		
		Log.w( TAG, "NotificationService Started" );
	}

	private void shutdownLoggerService()
	{
		unregisterReceiver( broadcastReceiver );
		
		Log.w( TAG, "NotificationService Ended" );
	}

	@Override
	public int onStartCommand(Intent intent, int flags, int startId)
	{
		super.onStartCommand( intent, flags, startId );
		return START_STICKY;
	}

	@Override
	public void onCreate()
	{
		super.onCreate();
		
		startLoggerService();
	}

	@Override
	public void onDestroy()
	{
		super.onDestroy();

		shutdownLoggerService();
	}

	@Override
	public IBinder onBind( Intent intent )
	{
		return mBinder;
	}

	public void createVisit( String name, long when, long length )
	{
		Intent intent = new Intent(this, NotificationService.class);
		intent.setAction( ACTION_VISIT_START_NOTIFICATION );
		intent.putExtra( "name", name );
		intent.putExtra( "when", when );
		intent.putExtra( "length", length );
		onHandleIntent(intent);
	}
	
	public void visitEnded( String title, String name )
	{
		Intent intent = new Intent(this, NotificationService.class);
		intent.setAction( ACTION_VISIT_ENDED_NOTIFICATION );
		intent.putExtra( "title", title );
		intent.putExtra( "name", name );
		onHandleIntent(intent);
	}
	
	public void visitCancel()
	{
		Intent intent = new Intent(this, NotificationService.class);
		intent.setAction( ACTION_VISIT_CANCEL_NOTIFICATION );
		onHandleIntent(intent);
	}
	
	protected void visitEnd()
	{
		Intent intent = new Intent(this, NotificationService.class);
		intent.setAction( ACTION_VISIT_END );
		onHandleIntent(intent);
	}

	private void setAlarm( int triggerInTime, PendingIntent pendingIntent )
	{
		int ALARM_TYPE = AlarmManager.RTC_WAKEUP;
		AlarmManager am = (AlarmManager) getSystemService(Context.ALARM_SERVICE);
		Calendar calendar = Calendar.getInstance();
		calendar.add(Calendar.SECOND, triggerInTime);
		
		if (Build.VERSION.SDK_INT >= Build.VERSION_CODES.M)
		{
			am.setExactAndAllowWhileIdle( ALARM_TYPE, calendar.getTimeInMillis(), pendingIntent );
		}
		else if ( Build.VERSION.SDK_INT >= Build.VERSION_CODES.LOLLIPOP )
		{
			am.setExact( ALARM_TYPE, calendar.getTimeInMillis(), pendingIntent );
		}
		else
		{
			am.set( ALARM_TYPE, calendar.getTimeInMillis(), pendingIntent );
		}

		Log.d( TAG, "Added alarm to trigger at " + new DateService().getTimeString( calendar.getTimeInMillis() / 1000L ) );
	}

	protected PendingIntent getVisitEndIntent()
	{
		Intent laterIntent = new Intent( ACTION_VISIT_END );
		return PendingIntent.getBroadcast( this, 0x0, laterIntent, PendingIntent.FLAG_UPDATE_CURRENT );
	}

	protected PendingIntent getAliveIntent()
	{
		Intent refreshIntent = new Intent( ACTION_VISIT_REFRESH );
		return PendingIntent.getBroadcast( this, 0x1, refreshIntent, PendingIntent.FLAG_UPDATE_CURRENT );
	}

	private void cancelAliveAlarm()
	{
		AlarmManager alarmMgr = (AlarmManager) getSystemService( Context.ALARM_SERVICE );
		alarmMgr.cancel( getAliveIntent() );
	}

	protected void onHandleIntent( Intent intent )
	{
		if ( intent != null )
		{
			final String action = intent.getAction();
			
			if ( ACTION_VISIT_START_NOTIFICATION.equals( action ) )
			{
				Log.d( TAG, "Creating Visit Start Notification" );
				
				long when = intent.getLongExtra( "when", 0 );
				long len = intent.getLongExtra( "length", 0 );
				long alreadyElapsed = ( System.currentTimeMillis() - when ) / 1000L;
				int alarmLength = (int)( len - alreadyElapsed );
				Log.d( TAG, "Alarm set to trigger in " + alarmLength + " secs. ( when: " + when + " length: " + len + ")" );
				
				if ( alarmLength > 0 )
				{
					Notification notification = new NotificationBuilderService().createVisitNotification( this, intent.getStringExtra( "name" ), when );

					startForeground( NotificationBuilderService.VISIT_CARD_ID, notification );

					setAlarm( alarmLength, getVisitEndIntent() );

					setAlarm( ALARM_REFRESH_SECS, getAliveIntent() );
				}
			}
			else if ( ACTION_VISIT_ENDED_NOTIFICATION.equals( action ) )
			{
				Log.d( TAG, "Creating Visit Ended Notification" );
				
				new NotificationBuilderService().sendVisitEnded( this, NotificationBuilderService.VISIT_ENDED_ID, intent.getStringExtra( "title" ), intent.getStringExtra( "name" ) );
				
				stopForeground( true );
			}
			else if ( ACTION_VISIT_CANCEL_NOTIFICATION.equals( action ) )
			{
				Log.d( TAG, "Canceling Visit Notification" );
				
				new NotificationBuilderService().cancel( this, NotificationBuilderService.VISIT_CARD_ID );

				stopForeground( true );
			}
			else if ( ACTION_VISIT_END.equals( action ) )
			{
				Log.d( TAG, "End Visits intent triggered from Alarm." );

				cancelAliveAlarm();

				AlarmVisitEndRequest.addRequest();
				
				stopForeground( true );
			}
		}
	}
}