// SimpleFileDialog.java
package com.ensoft.ace.view.widget;

/*
* 
* This file is licensed under The Code Project Load License (CPOL) 1.02
* http://www.codeproject.com/info/cpol10.aspx
* http://www.codeproject.com/info/CPOL.zip
* 
* License Preamble:
* This License governs Your use of the Work. This License is intended to allow developers to use the Source
* Code and Executable Files provided as part of the Work in any application in any form.
* 
* The main points subject to the terms of the License are:
*    Source Code and Executable Files can be used in commercial applications;
*    Source Code and Executable Files can be redistributed; and
*    Source Code can be modified to create derivative works.
*    No claim of suitability, guarantee, or any warranty whatsoever is provided. The software is provided "as-is".
*    The Article(s) accompanying the Work may not be distributed or republished without the Author's consent
* 
* This License is entered between You, the individual or other entity reading or otherwise making use of
* the Work licensed pursuant to this License and the individual or other entity which offers the Work
* under the terms of this License ("Author").
*  (See Links above for full license text)
*/

import android.app.AlertDialog;
import android.content.Context;
import android.content.DialogInterface;
import android.content.DialogInterface.OnClickListener;
import android.text.Editable;
import android.view.Gravity;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewGroup.LayoutParams;
import android.widget.ArrayAdapter;
import android.widget.Button;
import android.widget.EditText;
import android.widget.LinearLayout;
import android.widget.TextView;
import android.widget.Toast;

import com.ensoft.ace.infrastructure.service.FileService;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;

public class SimpleFileDialog
{
	public String DEFAULT_FILE_NAME = "default.txt";
	private int FileOpen = 0;
	private int FileSave = 1;
	private int FolderChoose = 2;
	private int SelectType = FileSave;
	private String baseDirectory = "";
	private Context context;
	private TextView titleView1;
	private TextView titleView;
	private String selectedFileName = DEFAULT_FILE_NAME;
	private EditText inputText;

	private String dir = "";
	private List<String> subdirs = null;
	private SimpleFileDialogListener simpleFileDialogListener = null;
	private ArrayAdapter<String> listAdapter = null;

	public SimpleFileDialog( Context context, String fileSelectType, SimpleFileDialogListener simpleFileDialogListener )
	{
		this( context, fileSelectType, simpleFileDialogListener, new FileService().getDefaultSaveDir() );
	}

	public SimpleFileDialog( Context context, String fileSelectType, SimpleFileDialogListener simpleFileDialogListener, String baseDirectory )
	{
		switch ( fileSelectType )
		{
			case "FileOpen":
				SelectType = FileOpen;
				break;
			case "FileSave":
				SelectType = FileSave;
				break;
			case "FolderChoose":
				SelectType = FolderChoose;
				break;
			default:
				SelectType = FileOpen;
				break;
		}

		this.context = context;

		File baseDirFile = new File( baseDirectory );

		this.baseDirectory = baseDirFile.getPath();

		baseDirFile.mkdirs();

		this.simpleFileDialogListener = simpleFileDialogListener;
	}

	public void chooseFileOrDir()
	{
		if ( dir.equals( "" ) ) chooseFileOrDir( baseDirectory );
		else chooseFileOrDir( dir );
	}

	public void chooseFileOrDir( String dir )
	{
		File dirFile = new File( dir );

		if ( !dirFile.exists() || !dirFile.isDirectory() )
		{
			dir = baseDirectory;
		}

		try
		{
			dir = new File( dir ).getCanonicalPath();
		}
		catch ( IOException ioe )
		{
			return;
		}

		this.dir = dir;
		subdirs = getDirectories( dir );

		class SimpleFileDialogOnClickListener implements OnClickListener
		{
			public void onClick( DialogInterface dialog, int item )
			{
				String m_dir_old = SimpleFileDialog.this.dir;
				String sel = "" + ( (AlertDialog) dialog ).getListView().getAdapter().getItem( item );
				if ( sel.charAt( sel.length() - 1 ) == '/' )
					sel = sel.substring( 0, sel.length() - 1 );

				// Navigate into the sub-directory
				if ( sel.equals( ".." ) )
				{
					SimpleFileDialog.this.dir = SimpleFileDialog.this.dir.substring( 0, SimpleFileDialog.this.dir.lastIndexOf( "/" ) );
				}
				else
				{
					SimpleFileDialog.this.dir += "/" + sel;
				}
				selectedFileName = DEFAULT_FILE_NAME;

				if ( ( new File( SimpleFileDialog.this.dir ).isFile() ) ) // If the selection is a regular file
				{
					SimpleFileDialog.this.dir = m_dir_old;
					selectedFileName = sel;
				}

				updateDirectory();
			}
		}

		AlertDialog.Builder dialogBuilder = createDirectoryChooserDialog( dir, subdirs,
				new SimpleFileDialogOnClickListener() );

		dialogBuilder.setPositiveButton( "OK", ( dialog, which ) -> {
			// Current directory chosen
			// Call registered listener supplied with the chosen directory
			if ( simpleFileDialogListener != null )
			{
				{
					if ( SelectType == FileOpen || SelectType == FileSave )
					{
						selectedFileName = inputText.getText() + "";
						simpleFileDialogListener.onChosenDir( SimpleFileDialog.this.dir + "/" + selectedFileName );
					}
					else
					{
						simpleFileDialogListener.onChosenDir( SimpleFileDialog.this.dir );
					}
				}
			}
		} ).setNegativeButton( "Cancelar", null );

		final AlertDialog dirsDialog = dialogBuilder.create();

		// Show directory chooser dialog
		dirsDialog.show();
	}

	private boolean createSubDir( String newDir )
	{
		File newDirFile = new File( newDir );
		if ( !newDirFile.exists() ) return newDirFile.mkdir();
		else return false;
	}

	private List<String> getDirectories( String dir )
	{
		List<String> dirs = new ArrayList<>();
		try
		{
			File dirFile = new File( dir );

			// if directory is not the base sd card directory add ".." for going up one directory
			if ( !this.dir.equals( baseDirectory ) ) dirs.add( ".." );

			if ( !dirFile.exists() || !dirFile.isDirectory() )
			{
				return dirs;
			}

			for ( File file : dirFile.listFiles() )
			{
				if ( file.isDirectory() )
				{
					// Add "/" to directory names to identify them in the list
					dirs.add( file.getName() + "/" );
				}
				else if ( SelectType == FileSave || SelectType == FileOpen )
				{
					// Add file names to the list if we are doing a file save or file open operation
					dirs.add( file.getName() );
				}
			}
		}
		catch ( Exception e )
		{
		}

		Collections.sort( dirs, ( o1, o2 ) -> o1.compareTo( o2 ) );
		return dirs;
	}

	private AlertDialog.Builder createDirectoryChooserDialog( String title, List<String> listItems,
															  OnClickListener onClickListener )
	{
		AlertDialog.Builder dialogBuilder = new AlertDialog.Builder( context );

		titleView1 = new TextView( context );
		titleView1.setLayoutParams( new LayoutParams( LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT ) );

		if ( SelectType == FileOpen ) titleView1.setText( "Abrir:" );
		if ( SelectType == FileSave ) titleView1.setText( "Guardar como:" );
		if ( SelectType == FolderChoose ) titleView1.setText( "Elegir carpeta:" );

		//need to make this a variable Save as, Load, Select Directory
		titleView1.setGravity( Gravity.CENTER_VERTICAL );
		titleView1.setBackgroundColor( -12303292 ); // dark gray 	-12303292
		titleView1.setTextColor( context.getResources().getColor( android.R.color.white ) );

		// Create custom view for AlertDialog title
		LinearLayout titleLayout1 = new LinearLayout( context );
		titleLayout1.setOrientation( LinearLayout.VERTICAL );
		titleLayout1.addView( titleView1 );

		if ( SelectType == FolderChoose || SelectType == FileSave )
		{
			Button newDirButton = new Button( context );
			newDirButton.setLayoutParams( new LayoutParams( LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT ) );
			newDirButton.setText( "Nueva carpeta" );
			newDirButton.setOnClickListener( v -> {
				final EditText input = new EditText( context );

				// Show new folder name input dialog
				new AlertDialog.Builder( context ).
						setTitle( "Nombre nueva carpeta" ).
						setView( input ).setPositiveButton( "OK", ( dialog, whichButton ) -> {
					Editable newDir = input.getText();
					String newDirName = newDir.toString();
					// Create new directory
					if ( createSubDir( dir + "/" + newDirName ) )
					{
						// Navigate into the new directory
						dir += "/" + newDirName;
						updateDirectory();
					}
					else
					{
						Toast.makeText( context, "Error al crear la carpeta '"
								+ newDirName + "'", Toast.LENGTH_SHORT ).show();
					}
				} ).setNegativeButton( "Cancelar", null ).show();
			}
										   );
			titleLayout1.addView( newDirButton );
		}

		LinearLayout titleLayout = new LinearLayout( context );
		titleLayout.setOrientation( LinearLayout.VERTICAL );

		titleView = new TextView( context );
		titleView.setLayoutParams( new LayoutParams( LayoutParams.MATCH_PARENT, LayoutParams.WRAP_CONTENT ) );
		titleView.setBackgroundColor( -12303292 ); // dark gray -12303292
		titleView.setTextColor( context.getResources().getColor( android.R.color.white ) );
		titleView.setGravity( Gravity.CENTER_VERTICAL );
		titleView.setText( title );

		titleLayout.addView( titleView );

		if ( SelectType == FileOpen || SelectType == FileSave )
		{
			inputText = new EditText( context );
			inputText.setText( DEFAULT_FILE_NAME );
			titleLayout.addView( inputText );
		}

		dialogBuilder.setView( titleLayout );
		dialogBuilder.setCustomTitle( titleLayout1 );
		listAdapter = createListAdapter( listItems );
		dialogBuilder.setSingleChoiceItems( listAdapter, -1, onClickListener );
		dialogBuilder.setCancelable( false );
		return dialogBuilder;
	}

	private void updateDirectory()
	{
		subdirs.clear();
		subdirs.addAll( getDirectories( dir ) );
		titleView.setText( dir );
		listAdapter.notifyDataSetChanged();

		if ( SelectType == FileSave || SelectType == FileOpen )
		{
			inputText.setText( selectedFileName );
		}
	}

	private ArrayAdapter<String> createListAdapter( List<String> items )
	{
		return new ArrayAdapter<String>( context, android.R.layout.select_dialog_item, android.R.id.text1, items )
		{
			@Override
			public View getView( int position, View convertView, ViewGroup parent )
			{
				View v = super.getView( position, convertView, parent );
				if ( v instanceof TextView )
				{
					// Enable list item (directory) text wrapping
					TextView tv = (TextView) v;
					tv.getLayoutParams().height = LayoutParams.WRAP_CONTENT;
					tv.setEllipsize( null );
				}
				return v;
			}
		};
	}

	public interface SimpleFileDialogListener
	{
		void onChosenDir( String chosenDir );
	}
} 
