package com.ensoft.ace.view.export;

import android.app.Activity;
import android.content.Intent;
import android.graphics.Bitmap;
import android.graphics.Canvas;
import android.graphics.Color;
import android.graphics.Paint;
import android.graphics.pdf.PdfDocument;
import android.net.Uri;
import android.print.PrintAttributes;
import android.print.pdf.PrintedPdfDocument;
import android.view.View;
import android.view.ViewGroup;
import android.view.ViewTreeObserver;
import android.widget.FrameLayout;
import android.widget.LinearLayout;
import android.widget.RelativeLayout;

import com.ensoft.ace.App;
import com.ensoft.ace.domain.model.Flowmetry;
import com.ensoft.ace.domain.model.PatientCard;
import com.ensoft.ace.infrastructure.service.BitmapService;
import com.ensoft.ace.infrastructure.service.FileService;
import com.ensoft.ace.infrastructure.service.GraphPrinterService;
import com.ensoft.ace.infrastructure.service.LiverpoolNomogramFemalesService;
import com.ensoft.ace.infrastructure.service.LiverpoolNomogramMalesService;
import com.ensoft.ace.infrastructure.service.LiverpoolNomogramService;
import com.ensoft.ace.infrastructure.service.MetricsService;
import com.jjoe64.graphview.DefaultLabelFormatter;
import com.jjoe64.graphview.GraphView;
import com.jjoe64.graphview.LegendRenderer;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.text.NumberFormat;
import java.util.Locale;

import androidx.core.content.FileProvider;

public class FlowmetryPdf
{
	private static int PIXEL_DENSITY = 2;
	private static int SHEET_WIDTH = 592 * PIXEL_DENSITY;
	private static int SHEET_HEIGHT = 840 * PIXEL_DENSITY;

	protected Activity activity;
	protected Flowmetry flowmetry;
	protected GraphView graphView = null;
	private PatientCard patientCard;
	private String report;
	private PdfDocument document;
	private PdfDocument.Page page;
	private Mode mode;
	private int numPages;

	public FlowmetryPdf( Activity activity, Flowmetry flowmetry, PatientCard patientCard, String report, Mode mode, int numPages )
	{
		this.activity = activity;
		this.flowmetry = flowmetry;
		this.patientCard = patientCard;
		this.report = report;
		this.mode = mode;
		this.numPages = numPages;

		if ( !flowmetry.foundResults() )
			flowmetry.findResults();

		if ( Mode.PRINT == mode )
		{
			PrintAttributes printAttrs = new PrintAttributes.Builder().
					setColorMode( PrintAttributes.COLOR_MODE_COLOR ).
					setMediaSize( PrintAttributes.MediaSize.NA_LETTER ).
					setResolution( new PrintAttributes.Resolution( "zooey", Activity.PRINT_SERVICE, SHEET_WIDTH, SHEET_HEIGHT ) ).
					setMinMargins( PrintAttributes.Margins.NO_MARGINS ).
					build();

			document = new PrintedPdfDocument( this.activity, printAttrs );
		}
		else
		{
			document = new PdfDocument();
		}

		PdfDocument.PageInfo pageInfo = new PdfDocument.PageInfo.Builder( SHEET_WIDTH, SHEET_HEIGHT, 1 ).create();

		page = document.startPage( pageInfo );

		drawPdf( page.getCanvas() );
	}

	private static String ppFloat( float val )
	{
		return String.format( Locale.getDefault(), "%.2f", val );
	}

	private void drawPdfFinal( final Canvas canvas, View graphLayout )
	{
		Paint paint = new Paint();

		// 595x842
		// units are in points (1/72 of an inch)
		int sheetW = SHEET_WIDTH;
		int sheetH = SHEET_HEIGHT;
		int titleBaseLine = MetricsService.cmToPs( 5 * PIXEL_DENSITY );
		int oneRow = 20 * PIXEL_DENSITY;
		int leftMargin = 54 * PIXEL_DENSITY; // ~2cm
		int firstCol = leftMargin;
		int secCol = leftMargin + sheetW / 2 - leftMargin / 2 + MetricsService.cmToPs( PIXEL_DENSITY ); // half sheet second collumn
		int resFirstColPos = firstCol + MetricsService.cmToPs( 4 * PIXEL_DENSITY );
		int resSecondColPos = secCol + MetricsService.cmToPs( 5 * PIXEL_DENSITY );
		int lastY = titleBaseLine;

		Bitmap bmp = BitmapService.getBitmapFromView( graphLayout );

		canvas.drawBitmap( bmp, leftMargin / 2, lastY - MetricsService.cmToPs( PIXEL_DENSITY ), paint );
		( (ViewGroup) graphLayout.getParent() ).removeView( graphLayout );

		paint.setColor( Color.BLACK );
		paint.setTextSize( 12 * PIXEL_DENSITY );
		paint.setFakeBoldText( true );
		canvas.drawText( "Resultados:", secCol, lastY, paint );
		paint.setFakeBoldText( false );

		lastY += oneRow;
		paint.setColor( Color.RED );
		canvas.drawText( "Uroflujometría:", secCol, lastY, paint );

		paint.setColor( Color.BLACK );

		lastY += oneRow;
		canvas.drawText( "Q Máximo:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getQMax() ) + " ml/s", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Volumen Total:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getTotalVol() ) + " ml", resSecondColPos, lastY, paint ); // i need the value in ml

		lastY += oneRow;
		canvas.drawText( "Volumen hasta Q Max:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getVolAtQMaxTime() ) + " ml", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Q Promedio:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getQAverage() ) + " ml/s", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Tiempo Total:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getTotalTime() ) + " seg", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Q a 2 seg:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getQAt2Seconds() ) + " ml/s", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Tiempo Neto:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getNetTime() ) + " seg", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "T a QMax:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getQMaxTime() ) + " seg", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Tiempo de descenso:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getTotalDescTime() ) + " seg", resSecondColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Tiempo entre pausas:", secCol, lastY, paint );
		canvas.drawText( ppFloat( flowmetry.getTimePaused() ) + " seg", resSecondColPos, lastY, paint );

		int secHalf = SHEET_HEIGHT / 2 + MetricsService.cmToPs( 3 * PIXEL_DENSITY );
		lastY = secHalf - 50 * PIXEL_DENSITY;
		paint.setFakeBoldText( true );
		canvas.drawText( "Ficha:", firstCol, lastY, paint );
		paint.setFakeBoldText( false );

		lastY += oneRow;
		canvas.drawText( "Datos del Paciente:", firstCol, lastY, paint );

		lastY += oneRow * 2;
		canvas.drawText( "Nombre:", firstCol, lastY, paint );
		canvas.drawText( patientCard.getFullName(), resFirstColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Ciudad:", firstCol, lastY, paint );
		canvas.drawText( patientCard.getCity(), resFirstColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Teléfono:", firstCol, lastY, paint );
		canvas.drawText( patientCard.getPhoneNumber(), resFirstColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Mutual:", firstCol, lastY, paint );
		canvas.drawText( patientCard.getMedicalInsurance(), resFirstColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Edad:", firstCol, lastY, paint );
		canvas.drawText( String.valueOf( patientCard.getAge() ), resFirstColPos, lastY, paint );

		lastY += oneRow;
		canvas.drawText( "Sexo:", firstCol, lastY, paint );
		canvas.drawText( patientCard.getGender(), resFirstColPos, lastY, paint );

		/* Informe */
		lastY += oneRow * 2;
		paint.setColor( Color.BLACK );
		paint.setTextSize( 12 * PIXEL_DENSITY );
		paint.setFakeBoldText( true );
		canvas.drawText( "Informe:", firstCol, lastY, paint );
		paint.setFakeBoldText( false );
		lastY += oneRow;
		for ( String line : report.split( "\n" ) )
		{
			canvas.drawText( line, firstCol, lastY, paint );
			lastY += paint.descent() - paint.ascent();
		}

		lastY = secHalf - 50 * PIXEL_DENSITY;
		paint.setFakeBoldText( true );
		canvas.drawText( "Datos del Estudio:", secCol, lastY, paint );
		paint.setFakeBoldText( false );

		lastY += oneRow * 2;
		canvas.drawText( "Fecha:", secCol, lastY, paint );
		canvas.drawText( patientCard.getFullDateString(), resSecondColPos, lastY, paint );

		document.finishPage( page );

		if ( numPages >= 2 )
		{
			drawPage2();
		}
		else
		{
			endDoc();
		}
	}

	private void drawPage2()
	{
		PdfDocument.PageInfo pageInfo = new PdfDocument.PageInfo.Builder( SHEET_WIDTH, SHEET_HEIGHT, 1 ).create();

		page = document.startPage( pageInfo );

		GraphPrinterService.drawLiverpoolGraph( activity, flowmetry, LinearLayout.HORIZONTAL, patientCard, ( graphLayout, liverpoolNomogramService ) -> drawPdfFinal2( page.getCanvas(), graphLayout, liverpoolNomogramService ) );
	}

	private void drawPdfFinal2( final Canvas canvas2, View graphLayout2, LiverpoolNomogramService liverpoolNomogramService )
	{
		int titleBaseLine = MetricsService.cmToPs( 5 * PIXEL_DENSITY );
		int oneRow = 20 * PIXEL_DENSITY;
		int leftMargin = 54 * PIXEL_DENSITY; // ~2cm
		int firstCol = leftMargin;
		int lastY = SHEET_WIDTH / 2 + titleBaseLine + oneRow;

		Paint paint = new Paint();

		Bitmap bmp = BitmapService.getBitmapFromView( liverpoolNomogramService.getVolMaxGraph() );
		Bitmap bmp2 = BitmapService.getBitmapFromView( liverpoolNomogramService.getAverageGraph() );

		canvas2.drawBitmap( bmp, leftMargin / 2, titleBaseLine, paint );

		canvas2.drawBitmap( bmp2, leftMargin / 2 + bmp.getWidth(), titleBaseLine, paint );

		( (ViewGroup) graphLayout2.getParent() ).removeView( graphLayout2 );

		/** Observaciones */
		lastY += oneRow * 2;
		paint.setColor( Color.BLACK );
		paint.setTextSize( 12 * PIXEL_DENSITY );
		paint.setFakeBoldText( true );
		canvas2.drawText( "Observaciones:", firstCol, lastY, paint );
		lastY += oneRow;
		int tY = lastY;
		for ( String line : patientCard.getObservations().split( "\n" ) )
		{
			canvas2.drawText( line, firstCol, tY, paint );
			tY += paint.descent() - paint.ascent();
		}

		document.finishPage( page );

		endDoc();
	}

	private void endDoc()
	{
		try
		{
			File pdfDirPath = new File( new FileService().getPdfSaveDir() );
			pdfDirPath.mkdirs();

			File file = new File( pdfDirPath,
					"Flujometría - " + patientCard.getFullName() + " - " + patientCard.getNiceDateString() + ".pdf"
			);

			OutputStream os = new FileOutputStream( file );
			document.writeTo( os );
			document.close();
			os.close();

			if ( mode == Mode.SHARE )
			{
				shareDocument( FileProvider.getUriForFile( App.instance, "com.ensoft.ace",  file ) );
			}
			else if ( mode == Mode.PRINT )
			{
				// Print the thing
			}
		}
		catch ( IOException e )
		{
			throw new RuntimeException( "Error generating file", e );
		}
	}

	private void drawPdf( final Canvas canvas )
	{
		GraphPrinterService.drawFlowmetryGraph( activity, flowmetry, ( graphLayout ) -> drawPdfFinal( canvas, graphLayout ) );
	}

	private void shareDocument( Uri uri )
	{
		Intent shareIntent = new Intent();
		shareIntent.setAction( Intent.ACTION_SEND );
		shareIntent.setType( "application/pdf" );

		// Assuming it may go via eMail:
		shareIntent.putExtra( Intent.EXTRA_SUBJECT, "Flujometría de paciente: " + patientCard.getFullName() );
		shareIntent.putExtra( Intent.EXTRA_TEXT, "Se adjunta estudio de flujometría del paciente " + patientCard.getFullName() );

		// Attach the PDf as a Uri, since Android can't take it as bytes yet.
		shareIntent.putExtra( Intent.EXTRA_STREAM, uri );

		activity.startActivity( shareIntent );
	}

	public enum Mode
	{
		SHARE,
		PRINT,
		SAVE
	}
}
