package com.ensoft.ace.view.activity;

import android.annotation.SuppressLint;
import android.bluetooth.BluetoothAdapter;
import android.bluetooth.BluetoothDevice;
import android.content.Intent;
import android.content.pm.ActivityInfo;
import android.content.res.Configuration;
import android.graphics.Color;
import android.os.Bundle;
import android.os.Handler;
import android.os.Handler.Callback;

import android.os.Looper;
import android.util.Log;
import android.view.Menu;
import android.view.MenuItem;
import android.view.View;
import android.widget.Button;
import android.widget.LinearLayout;
import android.widget.TextView;

import com.ensoft.ace.R;
import com.ensoft.ace.domain.model.Flowmetry;
import com.ensoft.ace.domain.service.emitter.Emitter;
import com.ensoft.ace.domain.service.emitter.EmitterBluetooth;
import com.ensoft.ace.domain.service.flowmetry.FlowmetryAnalysis;
import com.ensoft.ace.domain.service.graphview.DateXAsSecsLabelFormatter;
import com.ensoft.ace.infrastructure.listener.BluetoothDeviceStatusListener;
import com.ensoft.ace.infrastructure.listener.EmitterConnectionResultListener;
import com.ensoft.ace.infrastructure.service.BluetoothDeviceFinderService;
import com.ensoft.ace.infrastructure.service.PreferenceService;
import com.ensoft.ace.view.widget.BatteryView;
import com.jjoe64.graphview.GraphView;
import com.jjoe64.graphview.LegendRenderer;
import com.jjoe64.graphview.series.DataPoint;
import com.jjoe64.graphview.series.LineGraphSeries;

import java.text.NumberFormat;

import butterknife.BindView;
import butterknife.ButterKnife;

public class FlowmetryCaptureActivity extends BaseActivity implements BluetoothDeviceStatusListener, EmitterConnectionResultListener, FlowmetryAnalysis.Event
{
	protected EmitterBluetooth emitter = null;
	protected FlowmetryAnalysis flowmetry = null;
	protected GraphView graphView = null;
	protected LineGraphSeries<DataPoint> graphVolume = null;
	protected LineGraphSeries<DataPoint> graphFlow = null;
	protected boolean loadedData = false;
	protected int dataPos = 0;

	@BindView( R.id.start_but )
	protected Button startButton;

	@BindView( R.id.end_but )
	protected Button endButton;

	@BindView( R.id.battery_status_container )
	protected View batteryStatusCont;

	@BindView( R.id.battery_status_obtaining)
	protected TextView batteryTextView;

	@BindView( R.id.battery_status )
	protected BatteryView batteryView;

	protected BluetoothDeviceFinderService finderService = null;
	protected BluetoothDevice bluetoothDevice = null;

	protected boolean batteryStatusShown = true;
	
	protected final Callback callback = msg -> {
		if ( null != flowmetry && flowmetry.running() )
		{
			dataPos += 1d;

			if ( flowmetry.push( msg.arg1 ) )
			{
				graphVolume.appendData( new DataPoint( dataPos, flowmetry.currentVol ), true, 1000 );
				graphFlow.appendData( new DataPoint( dataPos, flowmetry.currentFlow ), true, 1000 );
				graphView.getViewport().scrollToEnd();

				if ( flowmetry.currentFlow >= 50 )
					graphView.getViewport().setMaxY( flowmetry.currentFlow + 5 );
				else if ( flowmetry.currentFlow >= 30 )
					graphView.getViewport().setMaxY( 50 );

				Log.d( "Sampleo", "Se inserta flujo: " + flowmetry.currentFlow + " y volumen: " + flowmetry.currentVol );
			}
			else
			{
				Log.d( "Sampleo", "Se ignora: " + msg.arg1 );
			}
		}

		return true;
	};

	@Override
	protected void onCreate( Bundle savedInstanceState )
	{
		super.onCreate( savedInstanceState );
		setContentView( R.layout.activity_flowmetry );
		ButterKnife.bind( this );
		
		keepScreenOn();

		Intent intent = getIntent();

		initGraph( intent.getBooleanExtra( "scrollView", true ) );

		int[] loadFlow = intent.getIntArrayExtra( "flowHist" );
		int[] loadVol = intent.getIntArrayExtra( "volumHist" );

		if ( null != loadFlow && loadFlow.length > 0 && null != loadVol && loadVol.length > 0 )
		{
			int timeInit = intent.getIntExtra( "timeInit", 0 );
			int timeEnd = intent.getIntExtra( "timeEnd", loadFlow.length );

			if ( -1 != timeEnd && -1 != timeInit )
			{
				loadData( timeInit, timeEnd - timeInit, loadFlow, loadVol );

				startButton.setVisibility( View.INVISIBLE );
				endButton.setText( R.string.quit );
			}
			else
			{
				// This should never happen, it only happens if the flowmetry never recorded data
				finish();
			}
		}
		else
		{
			finderService = new BluetoothDeviceFinderService( this, this, new PreferenceService().getBluetoothName() );

			if ( new PreferenceService().getShowBattery() )
			{
				if ( null == savedInstanceState )
				{
					batteryStatusCont.setVisibility( View.VISIBLE );
					batteryStatusShown = false;
				}
				else
				{
					batteryStatusCont.setVisibility( View.GONE );
					batteryStatusShown = true;
				}
			}
		}
	}

	protected void loadData( int init, int size, int[] flow, int[] vol )
	{
		loadedData = true;
		dataPos = 0;

		DataPoint[] flowDP = new DataPoint[size+2];
		DataPoint[] volDP = new DataPoint[size+2];

		dataPos += 1d;
		flowDP[0] = new DataPoint( dataPos, 0 );
		volDP[0] = new DataPoint( dataPos, 0 );
		
		int pos = init;
		int i;
		float maxY = 30;
		for ( i = 1; i <= size; i++ )
		{
			pos = init + i;
			dataPos += 1d;

			flowDP[i] = new DataPoint( dataPos, flow[pos] / 5.f );
			volDP[i] = new DataPoint( dataPos, vol[pos] / 4.f );
			
			float f = flow[i] / 5.f;
			if ( f > 30 && f < 50 ) {
				maxY = 50;
			} else if ( f > 50 ) {
				maxY = f + 5;
			}
		}
		
		dataPos += 1d;
		flowDP[i] = new DataPoint( dataPos, 0 );
		volDP[i] = new DataPoint( dataPos, vol[pos] / 4.f );
		
		graphVolume.resetData( volDP );
		graphFlow.resetData( flowDP );

		int perMin = ( 1000 / new PreferenceService().getLatency() ) * 60;
		int maxRoundedX = ( (int) Math.floor( size / perMin ) + 1 ) * perMin;

		graphView.getViewport().setXAxisBoundsManual( true );
		graphView.getViewport().setMinX( 0 );
		graphView.getViewport().setMaxX( maxRoundedX );
		graphView.getViewport().setMaxY( maxY );
	}

	protected void initGraph( boolean scrollView )
	{
		graphVolume = new LineGraphSeries<>();
		graphVolume.setTitle( "Volumen ( ml )" );
		graphVolume.setColor( Color.BLUE );

		graphFlow = new LineGraphSeries<>();
		graphFlow.setTitle( "Flujo ( ml/s)" );
		graphFlow.setColor( Color.RED );

		graphView = new GraphView( this );
		graphView.getViewport().setScrollable( true );

		if ( scrollView )
		{
			graphView.getViewport().setXAxisBoundsManual( true );
			graphView.getViewport().setMinX( 0 );
			graphView.getViewport().setMaxX( 100 );
		}

		graphView.setBackgroundColor( 0x000000 );

		LegendRenderer lr = graphView.getLegendRenderer();
		lr.setVisible( true );
		lr.setAlign( LegendRenderer.LegendAlign.TOP );
		lr.setBackgroundColor( Color.argb( 100, 100, 100, 100 ) );

		graphView.addSeries( graphFlow ); // data
		graphView.getViewport().setYAxisBoundsManual( true );
		graphView.getViewport().setMinY( 0 );
		graphView.getViewport().setMaxY( 30 );
		NumberFormat nf = NumberFormat.getInstance();
		nf.setMaximumFractionDigits( 0 );
		nf.setMinimumFractionDigits( 0 );
		graphView.getGridLabelRenderer().setLabelFormatter( new DateXAsSecsLabelFormatter( nf, nf ) );
		graphView.getGridLabelRenderer().setNumVerticalLabels( 6 );
		graphView.getGridLabelRenderer().setNumHorizontalLabels( 6 );

		graphView.getSecondScale().addSeries( graphVolume );
		graphView.getSecondScale().setMinY( 0 );
		graphView.getSecondScale().setMaxY( 1000 );

		LinearLayout layout = findViewById( R.id.graph1 );
		layout.addView( graphView );
	}

	@SuppressLint("SourceLockedOrientationActivity")
	protected void lockOrientation()
	{
		int currentOrientation = getResources().getConfiguration().orientation;

		if ( currentOrientation == Configuration.ORIENTATION_LANDSCAPE )
		{
			setRequestedOrientation( ActivityInfo.SCREEN_ORIENTATION_SENSOR_LANDSCAPE );
		}
		else
		{
			setRequestedOrientation( ActivityInfo.SCREEN_ORIENTATION_SENSOR_PORTRAIT );
		}
	}

	protected void startClick()
	{
		lockOrientation();

		dataPos = 0;
		graphVolume.resetData( new DataPoint[]{} );
		graphFlow.resetData( new DataPoint[]{} );

		flowmetry = new FlowmetryAnalysis( FlowmetryAnalysis.Mode.Automatic, new PreferenceService().getLatency(), this );
		flowmetry.start();

		if ( null == emitter )
		{
			emitterInit( bluetoothDevice );
		}
		else
		{
			emitter.setHandlerCallback( callback );
			emitter.startWorkerThread();
		}

		startButton.setText( R.string.pause );
	}

	protected void pauseClick()
	{
		if ( null != flowmetry && flowmetry.started() )
		{
			if ( flowmetry.paused() )
			{
				flowmetry.start();

				startButton.setText( R.string.pause );
			}
			else
			{
				startButton.setText( R.string.reinit );

				flowmetry.pause();
			}
		}
	}

	protected void showFinalResult()
	{
		if ( null != flowmetry )
		{
			Flowmetry fm = flowmetry.getFlowmetryData();

			loadData( fm.getTimeInit(), fm.getTimeEnd() - fm.getTimeInit(), fm.getFlowHist(), fm.getVolumHist() );

			startButton.setText( R.string.cancel );
			endButton.setText( R.string.accept );
		}
		else
		{
			finish();
		}
	}

	protected void acceptClick()
	{
		if ( null != flowmetry )
		{
			Intent intent = new Intent( this, FlowmetryResultActivity.class );
			Flowmetry fm = flowmetry.getFlowmetryData();
			intent.putExtra( "flowmetryData", fm );
			startActivity( intent );
		}

		finish();
	}

	protected void endClick()
	{
		if ( !loadedData )
		{
			if ( null != flowmetry )
			{
				flowmetry.end();
			}

			if ( null != emitter )
			{
				emitter.end();
			}

			showFinalResult();
		}
		else
		{
			finish();
		}
	}

	public void onClick( View v )
	{
		final int id = v.getId();

		switch ( id )
		{
			case R.id.start_but:
				if ( getString( R.string.cancel ).equals( startButton.getText() ) )
				{
					finish();
				}
				else
				{
					if ( null == flowmetry || !flowmetry.started() )
						startClick();
					else
						pauseClick();
				}

				break;
			case R.id.end_but:

				if ( getString( R.string.accept ).equals( endButton.getText() ) )
				{
					acceptClick();
				}
				else
				{
					endClick();
				}

				break;
		}
	}

	public void emitterInit( BluetoothDevice device )
	{
		if ( null != emitter )
		{
			emitter.end();
			emitter = null;
		}

		emitter = new EmitterBluetooth( device );
		emitter.init( this, callback, this );
	}
	
	@Override
	protected void onActivityResult(int requestCode, int resultCode, Intent data)
	{
		super.onActivityResult( requestCode, resultCode, data );
		if ( null != finderService )
			finderService.onActivityResult( requestCode, resultCode, data );
	}
	
	@Override
	protected void onDestroy()
	{
		if ( emitter != null )
		{
			emitter.end();
		}
		
		super.onDestroy();
	}

	@Override
	public boolean onCreateOptionsMenu( Menu menu )
	{
		getMenuInflater().inflate( R.menu.main, menu );
		return true;
	}

	protected void startConfig()
	{
		Intent settingsActivity = new Intent( getBaseContext(), PreferencesActivity.class );
		startActivity( settingsActivity );
	}

	@Override
	public boolean onOptionsItemSelected( MenuItem item )
	{
		int id = item.getItemId();

		if ( id == R.id.action_settings )
		{
			startConfig();
			return true;
		}

		return super.onOptionsItemSelected( item );
	}

	@Override
	public void onBackPressed()
	{
		if ( null == flowmetry || !flowmetry.running() )
		{
			super.onBackPressed();
		}
	}
	
	@Override
	public void onDeviceFound( BluetoothAdapter bluetoothAdapter, BluetoothDevice bluetoothDevice )
	{
		this.bluetoothDevice = bluetoothDevice;

		if ( new PreferenceService().getShowBattery() )
		{
			batteryStatusCont.setVisibility( View.VISIBLE );

			emitter = new EmitterBluetooth( bluetoothDevice );
			emitter.init( this, this );
		}
	}
	
	@Override
	public void onDeviceNotFound()
	{
		errorMessage( getString( R.string.bt_connection_error ), ( dialog, which ) -> finish() );
	}
	
	@Override
	public void onEmitterConnectionSuccess()
	{
		if ( !batteryStatusShown )
		{
			runOnUiThread( () -> {
				int battery = emitter.getCurrentBattery();
				if ( battery != Integer.MIN_VALUE )
				{
					batteryTextView.setText( R.string.battery_status );
					batteryView.setVisibility( View.VISIBLE );
					batteryView.setBatteryLevel( battery );
				}
				else
				{
					batteryTextView.setText( R.string.battery_error_2 );
					new PreferenceService().setShowBattery( false );
				}

				new Handler( Looper.getMainLooper() ).postDelayed( () -> batteryStatusCont.setVisibility( View.GONE ), 2000 );
			} );
		}
	}
	
	@Override
	public void onEmitterConnectionFail( String message )
	{
		onDeviceNotFound();
	}

	@Override
	public void onFlowmetryStart() {}

	@Override
	public void onFlowmetryEnd()
	{
		startButton.setVisibility( View.INVISIBLE );
	}
}
