package com.ensoft.ace.domain.service.emitter;

import android.app.Activity;
import android.bluetooth.BluetoothDevice;
import android.bluetooth.BluetoothSocket;
import android.os.AsyncTask;
import android.os.Handler;
import android.os.Looper;

import com.ensoft.ace.infrastructure.listener.EmitterConnectionResultListener;
import com.ensoft.ace.infrastructure.service.PreferenceService;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.util.UUID;

public class EmitterBluetooth extends Emitter
{
	private BluetoothSocket socket;
	private BluetoothDevice device = null;
	private OutputStream outputStream;
	private InputStream inputStream;
	
	public EmitterBluetooth( BluetoothDevice device )
	{
		this.device = device;
	}

	public static final byte GET_WEIGHT = 0x0;
	public static final byte GET_BATTERY = 0x2;
	public static final byte RESET_ROM = 0x3;

	public int sendValue( byte sendValue )
	{
		try
		{
			outputStream.write( sendValue );
			outputStream.flush();

			byte[] packetBytes = new byte[2];
			byte[] b = new byte[1];
			int i = 0;

			while ( i < 2 )
			{
				if ( inputStream.read( b ) == 1 )
				{
					packetBytes[i] = b[0];
					i++;
				}
			}

			return ( ( ( packetBytes[0] & 0xFF ) << 4 ) + ( ( packetBytes[1] & 0xFF ) >> 4 ) );
		}
		catch ( IOException e )
		{
			e.printStackTrace();
			return 0;
		}
	}

	public static int readInputStreamWithTimeout(InputStream is, byte[] b, int timeoutMillis) throws IOException
	{
		int bufferOffset = 0;
		long maxTimeMillis = System.currentTimeMillis() + timeoutMillis;
		while (System.currentTimeMillis() < maxTimeMillis && bufferOffset < b.length) {
			int readLength = java.lang.Math.min(is.available(),b.length-bufferOffset);
			// can alternatively use bufferedReader, guarded by isReady():
			int readResult = is.read(b, bufferOffset, readLength);
			if (readResult == -1) break;
			bufferOffset += readResult;
		}
		return bufferOffset;
	}

	public int sendValueWithTimeout( byte sendValue, int timeout )
	{
		try
		{
			outputStream.write( sendValue );
			outputStream.flush();

			byte[] packetBytes = new byte[2];
			byte[] b = new byte[1];
			int i = 0;

			while ( i < 2 )
			{
				int read = readInputStreamWithTimeout( inputStream, b, timeout );

				if ( read == 1 )
				{
					packetBytes[i] = b[0];
					i++;
				}
				else if ( read <= 0 )
				{
					return Integer.MIN_VALUE;
				}
			}

			return ( ( ( packetBytes[0] & 0xFF ) << 4 ) + ( ( packetBytes[1] & 0xFF ) >> 4 ) );
		}
		catch ( IOException e )
		{
			e.printStackTrace();
			return Integer.MIN_VALUE;
		}
	}
	
	@Override
	protected int getCurrentValue()
	{
		return sendValue( GET_WEIGHT );
	}

	public int getCurrentBattery()
	{
		int battery = sendValueWithTimeout( GET_BATTERY, 4000 );

		if ( Integer.MIN_VALUE == battery )
			return Integer.MIN_VALUE;

		if ( battery > 3330 )
		{
			return 100;
		}
		else if ( battery >= 3180 )
		{
			return 80;
		}
		else if ( battery >= 3080 )
		{
			return 60;
		}
		else if ( battery >= 3030 )
		{
			return 40;
		}
		else if ( battery >= 2890 )
		{
			return 20;
		}
		else
		{
			return 10;
		}
	}

	public void resetRom()
	{
		sendValueWithTimeout( RESET_ROM, 10 );
	}
	
	public void createBTSockets( boolean workerThread ) throws IOException
	{
		UUID uuid = UUID.fromString( "00001101-0000-1000-8000-00805f9b34fb" ); //Standard SerialPortService ID
		socket = device.createRfcommSocketToServiceRecord( uuid );
		socket.connect();
		outputStream = socket.getOutputStream();
		inputStream = socket.getInputStream();
		resetRom();

		if ( workerThread )
		{
			startWorkerThread();

			if ( null != emitterConnectionResultListener )
				emitterConnectionResultListener.onEmitterConnectionSuccess();
		}
		else
		{
			new Handler( Looper.getMainLooper() ).post( () -> {
				if ( null != EmitterBluetooth.this.emitterConnectionResultListener )
					EmitterBluetooth.this.emitterConnectionResultListener.onEmitterConnectionSuccess();
			} );
		}
	}

	public boolean init( Activity activity, EmitterConnectionResultListener emitterConnectionResultListener )
	{
		this.activity = activity;
		this.emitterConnectionResultListener = emitterConnectionResultListener;

		AsyncTask.execute( () -> {
			try
			{
				createBTSockets( false );
			}
			catch ( final Exception e )
			{
				new Handler( Looper.getMainLooper() ).post( () -> {
					if ( null != EmitterBluetooth.this.emitterConnectionResultListener )
						EmitterBluetooth.this.emitterConnectionResultListener.onEmitterConnectionFail( e.getMessage() );
				} );
			}
		} );

		return true;
	}

	@Override
	public boolean init( Activity activity, Handler.Callback DataReceivedCb, EmitterConnectionResultListener emitterConnectionResultListener )
	{
		super.init( activity, DataReceivedCb, emitterConnectionResultListener );
		
		AsyncTask.execute( () -> {
			try
			{
				createBTSockets( true );
			}
			catch ( final Exception e )
			{
				new Handler( Looper.getMainLooper() ).post( () -> {
					if ( null != EmitterBluetooth.this.emitterConnectionResultListener )
						EmitterBluetooth.this.emitterConnectionResultListener.onEmitterConnectionFail( e.getMessage() );
				} );
			}
		} );
		
		return true;
	}

	@Override
	public void end()
	{
		try
		{
			super.end();

			if ( null != workerThread )
			{
				workerThread.join();
			}

			if ( null != inputStream )
			{
				inputStream.close();
			}

			if ( null != outputStream )
			{
				outputStream.close();
			}

			if ( null != socket )
			{
				socket.close();
			}
		}
		catch ( Exception e )
		{
			e.printStackTrace();
		}
	}
}
